package systems.dennis.shared.servers.service;

import org.springframework.stereotype.Service;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.mongo.service.PaginationService;
import systems.dennis.shared.mongo.service.StringIdEntity;
import systems.dennis.shared.servers.exception.ServerException;
import systems.dennis.shared.servers.form.ServerConfigForm;
import systems.dennis.shared.servers.model.ServerConfig;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.servers.repository.ServerConfigRepo;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

@Service
@DataRetrieverDescription(repo = ServerConfigRepo.class, form = ServerConfigForm.class, model = ServerConfig.class)
public class ServerConfigService extends PaginationService<ServerConfig> {
    private Map<Object, ServerConfig> serverConfigs = new HashMap<>();

    public ServerConfigService(WebContext holder) {
        super(holder);
    }

    public void checkActiveTypeExists(ServerConfigForm element) {

        if (element.getActive() == null || !element.getActive()) {
            return;
        }

        if (Objects.equals(element.getType(), ServerTypeProvider.OTHER)) {
            return;
        }

        if (element.getId() == null) {
            if (count(getFilterImpl().eq("type", element.getType())
                    .and(getFilterImpl().eq("active", true)))> 0){
                throw new ServerException("server_already_exists");
            }
        } else {
            if (count(getFilterImpl().eq("type", element.getType()).and(getFilterImpl().eq("active", true))
                    .and(getFilterImpl().notEq(StringIdEntity.ID_FIELD, element.getId())))> 0) {
                throw new ServerException("server_already_exists");
            }
        }

    }

    @Override
    public ServerConfig afterAdd(ServerConfig object) {
        addToCache(object);

        return super.afterAdd(object);
    }

    @Override
    public void afterEdit(ServerConfig object, ServerConfig original) {
        addToCache(object);

        super.afterEdit(object, original);
    }

    @Override
    public ServerConfig afterDelete(ServerConfig object) {
        deleteFromCache(object);

        return super.afterDelete(object);
    }

    public ServerConfig findByServerConfigTypeName(String serverConfigTypeName) {
        ServerConfig fromCache = getFromCache(serverConfigTypeName);
        if (fromCache != null) {
            return fromCache;
        }

        var filter = getFilterImpl().eq("name", serverConfigTypeName  ).setJoinOn(ServerConfig.SERVER_CONFIG_FIELD)
                .and(getFilterImpl().eq("active", true));

        ServerConfig serverConfig = getRepository().filteredFirst(filter).orElse(null);

        if (serverConfig != null) {
            addToCache(serverConfig);
        }

        return serverConfig;
    }

    /**
     * Find first server withing the config type
     *
     * @param type   - a config type to look for types are : {@link ServerTypeProvider}
     * @param silent returns null of true, if false on null throws ItemNotFound exception
     * @return according to the #silent parameter returns server
     */

    public ServerConfig findByType(Long type, boolean silent) {

        ServerConfig fromCache = getFromCache(type);
        if (fromCache != null) {
            return fromCache;
        }

        var filter = getFilterImpl().eq("type", type).and(getFilterImpl().eq("active", true));


        ServerConfig res =getRepository().filteredFirst(filter).orElse(null);

        if (silent && res == null) {
            return res;
        }

        if (res == null) throw ItemNotFoundException.fromId(type);

        addToCache(res);

        return res;
    }

    private void addToCache(ServerConfig serverConfig) {
        Object key = getKey(serverConfig);

        if (serverConfig.getActive()) {
            serverConfigs.put(key, serverConfig);
        } else {
            serverConfigs.remove(key);
        }
    }

    private ServerConfig getFromCache(Object key) {
        return serverConfigs.get(key);
    }

    private void deleteFromCache(ServerConfig serverConfig) {
        Object key = getKey(serverConfig);
        serverConfigs.remove(key);
    }

    private Object getKey(ServerConfig serverConfig) {
        if (Objects.nonNull(serverConfig.getServerConfigType())) {
            return serverConfig.getServerConfigType().getName();
        } else {
            return serverConfig.getType();
        }
    }
}
