package systems.dennis.shared.postgres.repository.query_processors;

import jakarta.persistence.criteria.*;
import lombok.Data;
import systems.dennis.shared.postgres.repository.SpecificationFilter;
import systems.dennis.shared.repository.AbstractDataFilter;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Objects;


@Data
public abstract class AbstractClassProcessor {
    private AbstractDataFilter filter;
    private Root root;

    private Class<?> type = String.class;

    private Object parameter = null;

    public AbstractClassProcessor (AbstractDataFilter AbstractDataFilter, Root root){

        this.filter = AbstractDataFilter;
        this.root = root;

        if (AbstractDataFilter.getFieldClass() == null && AbstractDataFilter.getValue() != null){
            type = AbstractDataFilter.getValue().getClass();
        }

    }

    public boolean isNotNullCase() {
        return !filter.NOT_NULL_OPERATOR.equalsIgnoreCase(filter.getOperator())
                && !filter.NULL_OPERATOR.equalsIgnoreCase(filter.getOperator());
    }
    public void addToNullOrNotNullPredicate(CriteriaBuilder criteriaBuilder, Root root, List<Predicate> predicates) {
        if (Objects.equals(getFilter().getOperator(), filter.NULL_OPERATOR)) {
            predicates.add(criteriaBuilder.isNull(getPath(root)));
        } else {
            predicates.add(criteriaBuilder.isNotNull(getPath(root)));
        }
    }
    public  <T> Expression<T> getPath(Root root) {
        if (!getFilter().isComplex()) {
            return root.get(getFilter().getField());
        }

        String[] paths = getFilter().getOn().split("\\.");
        Join join = null;
        for (String j : paths) {
            if (join == null) {
                join = root.join(j);
            } else {
                join = join.join(j);
            }
        }

        return join.get(getFilter().getField());
    }

    public void processDefault( CriteriaBuilder criteriaBuilder,  List<Predicate> predicates) {

        Object value = getValue(filter.getValue());

        if (value == null) {
            throw new ArithmeticException("Item is wrong type");
        }
        if (filter.LESS_THEN.equalsIgnoreCase(filter.getOperator())) {
            predicates.add(criteriaBuilder.lessThan(getPath(root), (Date) value));
        }
        if (filter.MORE_THEN.equalsIgnoreCase(filter.getOperator())) {
            predicates.add(criteriaBuilder.greaterThan(getPath(root), (Date) value));
        }
        if (filter.LESS_EQUALS.equalsIgnoreCase(filter.getOperator())) {
            predicates.add(criteriaBuilder.lessThanOrEqualTo(getPath(root), (Date) value));
        }
        if (filter.MORE_EQUALS.equalsIgnoreCase(filter.getOperator())) {
            predicates.add(criteriaBuilder.greaterThanOrEqualTo(getPath(root), (Date) value));
        }


        if (filter.NOT_NULL_OPERATOR.equalsIgnoreCase(filter.getOperator())) {
            predicates.add(criteriaBuilder.isNotNull(getPath(root)));
        }
        if (filter.NULL_OPERATOR.equalsIgnoreCase(filter.getOperator())) {
            predicates.add(criteriaBuilder.isNull(getPath(root)));
        }
        if (filter.EQUALS_OPERATOR.equalsIgnoreCase(filter.getOperator()) && !String.class.isAssignableFrom(filter.getValue().getClass())) {
            predicates.add(criteriaBuilder.equal(getPath(root), value));
        }

        //There is a bug that excludes elements with a value of null from the search results
        if (filter.NOT_EQUALS_OPERATOR.equalsIgnoreCase(filter.getOperator())) {
            addIsNullOrNotEqualPredicate(criteriaBuilder, predicates, value);
        }
        if (filter.IN.equalsIgnoreCase(filter.getOperator())) {
            CriteriaBuilder.In inClause = criteriaBuilder.in(getPath(root));

            Collection cValue = (Collection) value;
            cValue.forEach((x)-> inClause.value(x) );
            predicates.add(inClause);
        }



    }

    protected void addIsNullOrNotEqualPredicate(CriteriaBuilder criteriaBuilder, List<Predicate> predicates, Object value) {
        predicates.add(
                criteriaBuilder.or(
                        criteriaBuilder.isNull(getPath(root)),
                        criteriaBuilder.notEqual(getPath(root), value)));
    }

    public abstract Object getValue(Object value);

    public static AbstractClassProcessor processor(SpecificationFilter fitler, Root root){
        if (fitler.getFieldClass() != null && Collection.class.isAssignableFrom(fitler.getFieldClass())){
            return new CollectionPredicateProcessor(fitler, root);
        }

        if (Date.class.equals(fitler.getFieldClass()) && !Date.class.isAssignableFrom(fitler.getValue().getClass())){
            return new DatePredicateProcessor(fitler, root);
        }

        if  (fitler.getFieldClass() != null && (Number.class.isAssignableFrom(fitler.getFieldClass()) || isPrimitiveNumber(fitler.getFieldClass()))) {
            return new NumberPredicateProcessor(fitler, root);
        }

        if (fitler.getFieldClass() != null && (String.class.isAssignableFrom(fitler.getFieldClass()))){
            return new StringPredicateProcessor(fitler, root);
        } else {
            return new SimplePredicateProcessor(fitler, root);
        }


    }

    private static boolean isPrimitiveNumber(Class c) {
        return int.class == c || long.class == c || short.class == c || double.class == c || float.class == c ;
    }

}
