package systems.dennis.shared.postgres.repository;

import jakarta.persistence.criteria.CriteriaBuilder;
import jakarta.persistence.criteria.CriteriaQuery;
import jakarta.persistence.criteria.Predicate;
import jakarta.persistence.criteria.Root;
import lombok.Data;
import org.springframework.data.jpa.domain.Specification;
import systems.dennis.shared.exceptions.StandardException;
import systems.dennis.shared.model.IDPresenter;
import systems.dennis.shared.postgres.repository.query_processors.AbstractClassProcessor;
import systems.dennis.shared.repository.AbstractDataFilter;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

@Data
public class SpecificationFilter<T extends IDPresenter<?>> implements AbstractDataFilter<T> {

    boolean empty = true;
    private Specification<T> root;
    private boolean complex;
    private String on;
    private Class<?> type;
    private boolean insensitive;

    private boolean closed;
    private String operationType;
    private Object value;
    private String field;

    @Override
    public SpecificationFilter<T> operator(String field, Object value, String type) {

        if (this.isClosed()){
            throw new StandardException("query_was_already_closed", "only add/or functions are now available. " +
                    "Please check that 'operation' is performed after additional parameter' ");
        }

        if (field != null && type != null){
            setEmpty(false);
        }

        this.field = field;
        this.value = value;
        this.operationType = type;

        root = new Specification<T>() {
            @Override
            public Predicate toPredicate(Root<T> root, CriteriaQuery<?> query, CriteriaBuilder criteriaBuilder) {
                var qq = AbstractClassProcessor.processor(SpecificationFilter.this, root);

                List<Predicate> predicates = new ArrayList<>();
                if (!qq.isNotNullCase()){
                    qq.addToNullOrNotNullPredicate(criteriaBuilder, root, predicates);
                } else {
                    qq.processDefault(criteriaBuilder, predicates);
                }

                return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
            }
        };

        this.closed = true;
        return this;
    }

    private Specification<T> getRoot(){
        return  root;
    }


    @Override
    public <E extends AbstractDataFilter<?>> E and(E filter) {
        if (root == null){
            throw new StandardException("query_has_still_open", "close query to use 'and' operator");
        }
        if (isEmpty(filter)){
            return (E) this;
        }
        SpecificationFilter<T> item = (SpecificationFilter<T>) filter;
        this.empty = false;
        root = root.and(item.getRoot());
        return (E) this;
    }

    @Override
    public <E extends AbstractDataFilter<?>> E or(E filter) {
        if (root == null) {
            throw new StandardException("query_has_still_open", "close query to use 'or' operator");
        }
        if (filter.isEmpty()){
            return (E) this;
        }
        SpecificationFilter<T> item = (SpecificationFilter<T>) filter;
        root = root.and(item.getRoot());
        this.empty = false;
        return (E) this;
    }

    @Override
    public <E extends AbstractDataFilter<T>> E comparasionType(Class<?> type) {
        this.type = type;
        return (E) this;
    }

    @Override
    public SpecificationFilter<T> setInsensitive(boolean insensitive) {
        this.insensitive = insensitive;
        return this;
    }

    @Override
    public SpecificationFilter<T> setComplex(boolean complex) {
        this.complex = complex;
        return this;
    }

    @Override
    public SpecificationFilter<T> setJoinOn(String on) {
        this.on = on;
        return this;
    }

    @Override
    public Serializable getIdValue(Object id) {
        return (long) id;
    }

    @Override
    public boolean isEmpty() {
        return empty;
    }

    @Override
    public String getOperator() {
        return operationType;
    }

    boolean isEmpty(AbstractDataFilter<?> filter){
        return  filter == null || filter.isEmpty();
    }

    public Class<?> getFieldClass() {
        try {

            return type == null ? getValue().getClass() : type;
        } catch (Exception e){
            return  String.class;
        }
    }

    @Override
    public <T1> T1 getQueryRoot() {
        return (T1) this.root;
    }

    public boolean getInsensitive() {
        return this.insensitive;
    }
}
