package systems.dennis.shared.mongo.repository;

import com.mongodb.BasicDBObject;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.bson.Document;
import org.bson.types.ObjectId;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationOperation;
import org.springframework.data.mongodb.core.aggregation.ArrayOperators;
import org.springframework.data.mongodb.core.aggregation.FacetOperation;
import org.springframework.data.mongodb.core.aggregation.MatchOperation;
import org.springframework.data.mongodb.core.mapping.DBRef;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.CriteriaDefinition;
import systems.dennis.shared.exceptions.StandardException;
import systems.dennis.shared.model.IDPresenter;
import systems.dennis.shared.mongo.exception.IncorrectSpecification;
import systems.dennis.shared.mongo.repository.query_processors.AbstractClassProcessor;
import systems.dennis.shared.repository.AbstractDataFilter;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@Data
@Slf4j
public class MongoSpecification<T extends IDPresenter<?>> implements AbstractDataFilter<T> {
    private static final String ID_FIELD = "id";
    public static final String OR_OPERATOR = "or";
    public static final String AND_OPERATOR = "and";

    private boolean empty;
    private boolean closed;
    private boolean insensitive;
    private boolean complex;

    private Class<?> type;


    private String operationType;
    private Object value;
    private String field;
    private String on;

    boolean calculated = false;

    private List<MongoSpecification<T>> or = new ArrayList<>();

    private List<MongoSpecification<T>> and = new ArrayList<>();

    private Criteria criteria;

    @Override
    public MongoSpecification<T> operator(String field, Object value, String type) {
        if (this.isClosed()) {
            throw new StandardException("query_was_already_closed", "only add/or functions are now available. " +
                    "Please check that 'operation' is performed after additional parameter' ");
        }

        if (field != null && type != null) {
            setEmpty(false);
        } else {
            setEmpty(true);
            return this;
        }

        if (isCalculated()) {
            throw new StandardException("query_was_already_closed", "due to limitations of Query you cannot use Query after you had already called method 'getCriteriaRoot()'");
        }

        this.field = field;
        this.value = value;
        this.operationType = type;


        criteria = Criteria.where(getField());
        var qq = AbstractClassProcessor.processor(this);
        if (!qq.isNotNullCase()) {
            qq.addToNullOrNotNullPredicate();
        } else {
            qq.processDefault();
        }
        this.closed = true;
        return this;

    }

    @Override
    public <E extends AbstractDataFilter<?>> E and(E filter) {

        checkCriteria();
        if (filter.isEmpty()) {

            return (E) this.copy((MongoSpecification )filter);
        }

        if (isCalculated()) {
            throw new StandardException("query_was_already_closed", "due to limitations of Query you cannot use Query after you had already called method 'getCriteriaRoot()'");
        }
        and.add((MongoSpecification<T>) filter);
        this.empty = false;
        return (E) this;

    }

    private void checkCriteria() {
        if (criteria == null){
            criteria = Criteria.where(field);
        }
    }

    private MongoSpecification copy(MongoSpecification filter) {
        this.criteria = filter.criteria;
        this.type = filter.type;
        this.and = filter.and;
        this.or = filter.or;
        this.on = filter.on;
        this.complex = filter.complex;
        this.insensitive = filter.insensitive;
        this.field = field;
        this.setEmpty(false);
        return this;
    }

    @Override
    public <E extends AbstractDataFilter<?>> E or(E filter) {
        if (isCalculated()) {
            throw new StandardException("query_was_already_closed", "due to limitations of Query you cannot use Query after you had already called method 'getCriteriaRoot()'");
        }

        if (filter.isEmpty()) {

            return (E) this.copy((MongoSpecification) filter);
        }

        or.add((MongoSpecification<T>) filter);

        this.empty = false;
        return (E) this;

    }

    @Override
    public <E extends AbstractDataFilter<T>> E comparasionType(Class<?> type) {
        this.type = type;
        return (E) this;
    }

    @Override
    public boolean isClosed() {
        return closed;
    }

    @Override
    public AbstractDataFilter<T> setInsensitive(boolean insensitive) {
        this.insensitive = insensitive;
        return this;
    }

    @Override
    public AbstractDataFilter<T> setComplex(boolean complex) {
        this.complex = complex;
        return this;
    }

    @Override
    public AbstractDataFilter<T> setJoinOn(String on) {
        this.on = on;
        return this;
    }

    public List<AggregationOperation> toAggregationOperations(Class<?> entityClass, MongoTemplate mongoTemplate) {
        List<AggregationOperation> operations = new ArrayList<>();
        Map<String, List<Criteria>> matchCriterias = addOperations(operations, entityClass, mongoTemplate, AND_OPERATOR);

        List<Criteria> orCriterias = matchCriterias.get(OR_OPERATOR);
        List<Criteria> andCriterias = matchCriterias.get(AND_OPERATOR);

        Criteria finalCriteria = new Criteria();

        if (!orCriterias.isEmpty() && !andCriterias.isEmpty()) {
            finalCriteria = new Criteria().orOperator(
                    new Criteria().andOperator(andCriterias.toArray(new Criteria[0])),
                    new Criteria().orOperator(orCriterias.toArray(new Criteria[0]))
            );
        } else if (!orCriterias.isEmpty()) {
            finalCriteria.orOperator(orCriterias.toArray(new Criteria[0]));
        } else if (!andCriterias.isEmpty()) {
            finalCriteria.andOperator(andCriterias.toArray(new Criteria[0]));
        }

        if (!finalCriteria.getCriteriaObject().isEmpty()) {
            operations.add(Aggregation.match(finalCriteria));
        }


        return operations;
    }

    private Map<String, List<Criteria>> addOperations(List<AggregationOperation> operations, Class<?> entityClass, MongoTemplate mongoTemplate, String operation) {
        Map<String, List<Criteria>> matchCriterias = new HashMap<>();
        matchCriterias.put(OR_OPERATOR, new ArrayList<>());
        matchCriterias.put(AND_OPERATOR, new ArrayList<>());

        Criteria matchCriteria;
        if (this.on != null) {
            matchCriteria = handleJoin(operations, entityClass, mongoTemplate);
        } else {
            matchCriteria = handleNoJoin(operations, entityClass, mongoTemplate);
        }

        if (matchCriteria != null) {
            matchCriterias.get(operation).add(matchCriteria);
        }

        if (!this.or.isEmpty()) {
            for (MongoSpecification<T> spec : this.or) {
                Map<String, List<Criteria>> orMatchCriterias = spec.addOperations(operations, entityClass, mongoTemplate, OR_OPERATOR);
                mergeCriterias(matchCriterias, orMatchCriterias);
            }
        }

        if (!this.and.isEmpty()) {
            for (MongoSpecification<T> spec : this.and) {
                Map<String, List<Criteria>> andMatchCriterias = spec.addOperations(operations, entityClass, mongoTemplate, AND_OPERATOR);
                mergeCriterias(matchCriterias, andMatchCriterias);
            }
        }

        return matchCriterias;
    }

    private void mergeCriterias(Map<String, List<Criteria>> target, Map<String, List<Criteria>> source) {
        for (Map.Entry<String, List<Criteria>> entry : source.entrySet()) {
            target.get(entry.getKey()).addAll(entry.getValue());
        }
    }

    public Boolean shouldAggregate(Class<?> entityClass) {
        if (this.on != null) {
            try {
                Class<?> joinClass = entityClass.getDeclaredField(this.on).getType();
                Field joinField = joinClass.getDeclaredField(this.field);
                if (joinField.isAnnotationPresent(DBRef.class)) {
                    return true;
                }
            } catch (Exception e) {
                throw new IncorrectSpecification("Error processing join fields");
            }
        }

        for (MongoSpecification<T> spec : this.or) {
            if (spec.shouldAggregate(entityClass)) {
                return true;
            }
        }

        for (MongoSpecification<T> spec : this.and) {
            if (spec.shouldAggregate(entityClass)) {
                return true;
            }
        }

        return false;
    }

    private Criteria handleJoin(List<AggregationOperation> operations, Class<?> entityClass, MongoTemplate mongoTemplate) {
        try {
            Class<?> joinClass = entityClass.getDeclaredField(this.on).getType();
            Field joinField = joinClass.getDeclaredField(this.field);
            String joinCollectionName = mongoTemplate.getCollectionName(joinClass);
            String rootCollectionName = mongoTemplate.getCollectionName(entityClass);
            String as = rootCollectionName + "_" + this.field;

            if (joinField.isAnnotationPresent(DBRef.class)) {
                String nestedCollectionName = mongoTemplate.getCollectionName(joinField.getType());
                return handleDbRefJoin(operations, joinCollectionName, rootCollectionName, nestedCollectionName);
            } else {
                return handleNonDbRefJoin(operations, joinCollectionName, rootCollectionName, as);
            }
        } catch (Exception e) {
            throw new IncorrectSpecification("Error processing join fields");
        }
    }

    private Criteria handleDbRefJoin(List<AggregationOperation> operations, String joinCollectionName, String rootCollectionName,
                                 String nestedCollectionName) {
        String value = ((T) this.criteria.getCriteriaObject().get(this.getField())).getId().toString();

        String asJoin = joinCollectionName + "_" + rootCollectionName;
        String asNested = joinCollectionName + "_" + nestedCollectionName;

        Criteria matchCriteria = Criteria.where(asNested + "._id").is(new ObjectId(value));

        operations.add(Aggregation.lookup(joinCollectionName, joinCollectionName + ".$id",  "_id", asJoin));
        operations.add(Aggregation.unwind(asJoin, true));
        operations.add(Aggregation.lookup(nestedCollectionName, asJoin + "." + this.field + ".$id",  "_id", asNested));
        operations.add(Aggregation.unwind(asNested, true));

        return matchCriteria;
    }

    private Criteria handleNonDbRefJoin(List<AggregationOperation> operations, String joinCollectionName, String rootCollectionName, String as) {
        Object value = this.criteria.getCriteriaObject().get(this.getField());
        Criteria matchCriteria = Criteria.where(as + "." + this.field).is(value);

        operations.add(Aggregation.lookup(joinCollectionName, this.on + ".$id", "_id", as));
        operations.add(Aggregation.unwind(as));

        return matchCriteria;
    }

    private Criteria handleNoJoin(List<AggregationOperation> operations, Class<?> entityClass, MongoTemplate mongoTemplate) {
        Field field;
        try {
            field = entityClass.getDeclaredField(this.field);
        } catch (Exception e) {
            field = null;
        }

        if (Objects.nonNull(field) && field.isAnnotationPresent(DBRef.class)) {
            return handleNoJoinDbRef(operations, entityClass, mongoTemplate);
        } else {
            return this.criteria;
        }
    }

    private Criteria handleNoJoinDbRef(List<AggregationOperation> operations, Class<?> entityClass, MongoTemplate mongoTemplate) {
        String rootCollectionName = mongoTemplate.getCollectionName(entityClass);
        String as = rootCollectionName + "_" + this.field;
        String value = ((T) this.criteria.getCriteriaObject().get(this.getField())).getId().toString();
        Criteria matchCriteria = Criteria.where(as + "._id").is(new ObjectId(value));

        operations.add(Aggregation.lookup(rootCollectionName, this.getField() + ".$id", "_id", as));

        return matchCriteria;
    }

    @Override
    public Serializable getIdValue(Object id) {
        return String.valueOf(id);
    }

    @Override
    public String getOperator() {
        return operationType;
    }

    @Override
    public Class<?> getFieldClass() {
        return type == null ? (value == null ? String.class : value.getClass()) : type ;
    }


    @Override
    public <E> E getQueryRoot() {

        if (calculated) return (E) criteria;

        for (MongoSpecification<T> mongoSpecification : or) {
            mongoSpecification.getQueryRoot();

        }

        if (!or.isEmpty())
            criteria.orOperator(or.stream().map(MongoSpecification::getCriteria).collect(Collectors.toList()));


        for (MongoSpecification<T> tMongoSpecification : and) {
            tMongoSpecification.getQueryRoot();

        }
        if (!and.isEmpty())
            criteria.andOperator(and.stream().map(MongoSpecification::getCriteria).collect(Collectors.toList()));
        calculated = true;

        return (E) criteria;
    }

    public Criteria getRoot() {
        return criteria;
    }


    public boolean getInsensitive() {
        return insensitive;
    }
}
