package systems.dennis.shared.mongo.repository;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.mongodb.core.query.Criteria;
import systems.dennis.shared.exceptions.StandardException;
import systems.dennis.shared.model.IDPresenter;
import systems.dennis.shared.mongo.repository.query_processors.AbstractClassProcessor;
import systems.dennis.shared.repository.AbstractDataFilter;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Data
@Slf4j
public class MongoSpecification<T extends IDPresenter<?>> implements AbstractDataFilter<T> {

    private boolean empty;
    private boolean closed;
    private boolean insensitive;
    private boolean complex;

    private Class<?> type;


    private String operationType;
    private Object value;
    private String field;
    private String on;

    boolean calculated = false;

    private List<MongoSpecification<T>> or = new ArrayList<>();

    private List<MongoSpecification<T>> and = new ArrayList<>();

    private Criteria criteria;

    @Override
    public MongoSpecification<T> operator(String field, Object value, String type) {
        if (this.isClosed()) {
            throw new StandardException("query_was_already_closed", "only add/or functions are now available. " +
                    "Please check that 'operation' is performed after additional parameter' ");
        }

        if (field != null && type != null) {
            setEmpty(false);
        } else {
            setEmpty(true);
            return this;
        }

        if (isCalculated()) {
            throw new StandardException("query_was_already_closed", "due to limitations of Query you cannot use Query after you had already called method 'getCriteriaRoot()'");
        }

        this.field = field;
        this.value = value;
        this.operationType = type;


        criteria = Criteria.where(getField());
        var qq = AbstractClassProcessor.processor(this);
        if (!qq.isNotNullCase()) {
            qq.addToNullOrNotNullPredicate();
        } else {
            qq.processDefault();
        }
        this.closed = true;
        return this;

    }

    @Override
    public <E extends AbstractDataFilter<?>> E and(E filter) {

        checkCriteria();
        if (filter.isEmpty()) {

            return (E) this.copy((MongoSpecification )filter);
        }

        if (isCalculated()) {
            throw new StandardException("query_was_already_closed", "due to limitations of Query you cannot use Query after you had already called method 'getCriteriaRoot()'");
        }
        and.add((MongoSpecification<T>) filter);
        this.empty = false;
        return (E) this;

    }

    private void checkCriteria() {
        if (criteria == null){
            criteria = Criteria.where(field);
        }
    }

    private MongoSpecification copy(MongoSpecification filter) {
        this.criteria = filter.criteria;
        this.type = filter.type;
        this.and = filter.and;
        this.or = filter.or;
        this.on = filter.on;
        this.complex = filter.complex;
        this.insensitive = filter.insensitive;
        this.field = field;
        this.setEmpty(false);
        return this;
    }

    @Override
    public <E extends AbstractDataFilter<?>> E or(E filter) {
        if (isCalculated()) {
            throw new StandardException("query_was_already_closed", "due to limitations of Query you cannot use Query after you had already called method 'getCriteriaRoot()'");
        }

        if (filter.isEmpty()) {

            return (E) this.copy((MongoSpecification) filter);
        }

        or.add((MongoSpecification<T>) filter);

        this.empty = false;
        return (E) this;

    }

    @Override
    public <E extends AbstractDataFilter<T>> E comparasionType(Class<?> type) {
        this.type = type;
        return (E) this;
    }

    @Override
    public boolean isClosed() {
        return closed;
    }

    @Override
    public AbstractDataFilter<T> setInsensitive(boolean insensitive) {
        this.insensitive = insensitive;
        return this;
    }

    @Override
    public AbstractDataFilter<T> setComplex(boolean complex) {
        this.complex = complex;
        return this;
    }

    @Override
    public AbstractDataFilter<T> setJoinOn(String on) {
        this.on = on;
        return this;
    }

    @Override
    public Serializable getIdValue(Object id) {
        return String.valueOf(id);
    }

    @Override
    public String getOperator() {
        return operationType;
    }

    @Override
    public Class<?> getFieldClass() {
        return type == null ? (value == null ? String.class : value.getClass()) : type ;
    }


    @Override
    public <E> E getQueryRoot() {

        if (calculated) return (E) criteria;

        for (MongoSpecification<T> mongoSpecification : or) {
            mongoSpecification.getQueryRoot();

        }

        if (!or.isEmpty())
            criteria.orOperator(or.stream().map(MongoSpecification::getCriteria).collect(Collectors.toList()));


        for (MongoSpecification<T> tMongoSpecification : and) {
            tMongoSpecification.getQueryRoot();

        }
        if (!and.isEmpty())
            criteria.andOperator(and.stream().map(MongoSpecification::getCriteria).collect(Collectors.toList()));
        calculated = true;

        return (E) criteria;
    }

    public Criteria getRoot() {
        return criteria;
    }


    public boolean getInsensitive() {
        return insensitive;
    }
}
