package systems.dennis.auth.role_validator;


import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.security.Keys;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import systems.dennis.auth.entity.ActiveToken;

import systems.dennis.auth.entity.Subscription;
import systems.dennis.auth.exception.LogoutException;
import systems.dennis.auth.repository.ActiveTokensRepo;

import systems.dennis.auth.repository.SubscriptionRepo;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.responses.SettingsProvider;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.ActiveTokenService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.utils.ApplicationContext;

import java.nio.charset.StandardCharsets;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Component
public class TokenProvider extends ApplicationContext {

    public static final String AUTHORITIES_KEY = "groups";
    public static final String USER_ID = "user_id";
    public static final String USER_EMAIL = "email";
    public static final String USER_LOGIN = "bid";
    public static final String USER_SCOPE = "s";
    public static final String USER_INNER_NUMBER = "userDataId";
    public static final String ISSUER = "usb";
    public static final String PURCHASES = "prchs";
    public static final String LANG = "lang";

    final
    UserDataRepository userRepository;


    final SubscriptionRepo subscriptionRepo;


    public TokenProvider(WebContext context, UserDataRepository userRepository,SubscriptionRepo subscriptionRepo) {
        super(context);
        this.userRepository = userRepository;

        this.subscriptionRepo = subscriptionRepo;
    }

    /**
     * Creates a token for user. If user not present it creates it
     *
     * @param dto      UserTokenDto is an information about user
     * @param roleList
     * @return String value with AUTHORITIES_KEY - list of ROLES, USER_ID - user ID from DB and signed with HS512 algorithm, with en expiration of  ${awiso.security.token.validity}
     */
    public ActiveToken createToken(UserTokenDTO dto, String tokenType, List<UserRole> roleList, ScopeModel scope) {
        ActiveTokensRepo repo = getBean(ActiveTokensRepo.class);

        var current = repo.findFirstByUserDataIdAndActiveIsTrueAndDueGreaterThanAndTypeAndScope(dto.getUserData().getId(), new Date(), tokenType, scope).orElse(null);

        if (current != null) return current;

        Calendar validity = Calendar.getInstance(); // creates calendar
        validity.setTime(new Date());

        validity.add(Calendar.MINUTE, getTokenValidity(scope));

        dto.setDue(validity.getTime());
        log.debug("We start Jwts Building...   {}, \n --> roles \n {} ", dto.getUserData(), roleList);


        var token = Jwts.builder()
                .setSubject(dto.getUserData().getEmail())
                .setIssuer(ISSUER)
                .claim(AUTHORITIES_KEY, roleList.stream().map(UserRole::getRole).collect(Collectors.toList()))
                .claim(USER_ID, dto.getUserData().getId())
                .claim(USER_EMAIL, dto.getUserData().getEmail())
                .claim(USER_LOGIN, dto.getUserData().getLogin())
                .claim(USER_SCOPE, scope.getName())

                .claim(LANG, getLanguage(dto))
                .signWith(Keys.hmacShaKeyFor(scope.getSecretKey().getBytes(StandardCharsets.UTF_8)))
                .setExpiration(validity.getTime())
                .compact();

        ActiveToken newToken = new ActiveToken();
        newToken.setToken(token);
        newToken.setActive(Boolean.TRUE);
        newToken.setDue(validity.getTime());
        newToken.setType(tokenType);
        newToken.setUserDataId(dto.getUserId());
        newToken.setScope(scope);
        return repo.save(newToken);

    }

    private String getLanguage(UserTokenDTO dto) {
        var service = getContext().getWebContext().getBean(SettingsProvider.class, null);
        return service == null ? "en" :  service.findUserSetting(dto.getUserId()).getLanguage();
    }




    public void removeAuthToken(String token, String type, ScopeModel scope) {
        ActiveTokenService repo = getBean(ActiveTokenService.class);
        AbstractDataFilter<ActiveToken> spec = getBean(ActiveTokenService.class).getFilterImpl().eq("token", token);
        spec = spec.and(getBean(ActiveTokenService.class).getFilterImpl().eq("scope", scope));
        spec = spec.and(getBean(ActiveTokenService.class).getFilterImpl().eq("type", type));
        ActiveToken item = repo.getRepository().filteredFirst(spec).orElseThrow(() -> new LogoutException("Token not exist for type: " + type));
        item.setActive(Boolean.FALSE);
        item.setClosed(new Date());
        repo.save(item);
    }

    private String getSecretKey() {
        return getContext().getEnv("dennis.systems.security.secret");
    }

    private int getTokenValidity(ScopeModel scope) {
        if (Objects.nonNull(scope.getScopeRule())) {
            Integer tokenDuration = scope.getScopeRule().getTokenDuration();
            if (tokenDuration != null && tokenDuration > 0) {
                return tokenDuration;
            }
        }
        return getContext().getEnv("dennis.systems.security.validity");
    }
}
