package systems.dennis.auth.controller;

import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.utils.AuthenticationService;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.AuthScopeService;
import systems.dennis.auth.service.ProfilePageService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.postgres.service.PaginationService;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.servers.model.ServerConfig;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.servers.repository.ServerConfigRepo;
import systems.dennis.shared.servers.service.ServerConfigService;
import systems.dennis.shared.utils.ApplicationContext;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import java.util.Hashtable;
import java.util.Optional;

;

@RequestMapping("api/v2/auth")
@RestController
@Slf4j
public class AuthorizeController extends ApplicationContext {


    private final AuthenticationService userService;


    public AuthorizeController(AuthenticationService userService, WebContext context) {
        super(context);
        this.userService = userService;

    }

    @SneakyThrows
    @PostMapping(path = "/login", produces = MediaType.APPLICATION_JSON_VALUE)
    public UserTokenDTO directLogin(@RequestBody LoginPassword password, @RequestParam(required = false) String backUrl, HttpServletRequest httpServletRequest) {
        Long userId = getBean(ProfilePageService.class).findByLogin(password.getLogin()).map(UserData::getId).orElseThrow(() -> ItemNotFoundException.fromId(password.getLogin()));
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(httpServletRequest, userId, false);
        return userService.authorize(password, scope);

    }

    @GetMapping("/test")

    @SneakyThrows
    public UserTokenDTO authLdap(@RequestBody LoginPassword loginPassword) {


        var cfg = getBean(ServerConfigService.class).getRepository().filteredFirst(getContext()
                .getDataFilterProvider().eq("active", true).and(getContext().getDataFilterProvider().eq("type", ServerTypeProvider.LDAP))).orElse(null);
        var config = (ServerConfig) cfg;
        if (config == null) {
            log.info(" NO LDAP CONFIG FOUND. return null");
            return null;
        }


        // set up the LDAP parameters
        Hashtable<Object, Object> env = new Hashtable<Object, Object>();
        env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
        env.put(Context.PROVIDER_URL, "ldap://" + config.getHost() + ":" + config.getPort());
        env.put(Context.SECURITY_AUTHENTICATION, "simple");
        env.put(Context.REFERRAL, "follow");
        env.put(Context.SECURITY_PRINCIPAL, config.getUserName());
        env.put(Context.SECURITY_CREDENTIALS, config.getPassword());

        // attempt to authenticate
        InitialLdapContext ctx = new InitialLdapContext(env, null);



        getUserInfo(loginPassword.getLogin(), ctx, getSearchControls(), config);


        ctx.close();

        return null;
    }


    private static UserData getUserInfo(String userName, InitialLdapContext ctx, SearchControls searchControls, ServerConfig config) {
        System.out.println("*** " + userName + " ***");
        UserData user = null;
        try {
            NamingEnumeration<SearchResult> answer = ctx.search(config.getServerParam(), "sAMAccountName=" + userName, searchControls);
            if (answer.hasMore()) {
                Attributes attrs = answer.next().getAttributes();
                System.out.println(attrs.get("distinguishedName"));
                System.out.println(attrs.get("givenname"));
                System.out.println(attrs.get("sn"));
                System.out.println(attrs.get("mail"));
                System.out.println(attrs.get("telephonenumber"));
                byte[] photo = (byte[])attrs.get("thumbnailPhoto").get();
                //savePhoto(userName, photo);
            } else {
                System.out.println("user not found.");
            }
            answer.close();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return user;
    }



    private static SearchControls getSearchControls() {
        SearchControls cons = new SearchControls();
        cons.setSearchScope(SearchControls.SUBTREE_SCOPE);
        String[] attrIDs = {"distinguishedName", "sn", "givenname", "mail", "telephonenumber", "thumbnailPhoto"};
        cons.setReturningAttributes(attrIDs);
        cons.setDerefLinkFlag(true);
        return cons;
    }


    private SearchControls getSimpleSearchControls() {
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        searchControls.setTimeLimit(13000);
        searchControls.setCountLimit(100);

        //String[] attrIDs = {"objectGUID"};
        //searchControls.setReturningAttributes(attrIDs);
        return searchControls;
    }

    @SneakyThrows
    @PostMapping(path = "/authorize_virtual_user", produces = MediaType.APPLICATION_JSON_VALUE)
    public UserTokenDTO virtualLogin(@RequestBody LoginPassword password) {
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(getContext().getRequest(), getCurrentUser(), true);
        return userService.authorizeVirtual(password, scope);

    }

    @SneakyThrows
    @PostMapping(path = "/restore/{login}", produces = MediaType.TEXT_HTML_VALUE)
    public String restore(@PathVariable("login") String login) {

        return userService.restore(login, getContext());
    }
}
