package systems.dennis.auth.client.required;

import io.jsonwebtoken.*;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.exception.ScopeException;
import systems.dennis.auth.exception.TokenExpiredException;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.entity.TokenData;
import systems.dennis.shared.utils.ApplicationContext;

import java.util.Date;
import java.util.List;

@Service
@Slf4j
public class TokenProviderClient extends ApplicationContext {


    public static final String AUTHORITIES_KEY = "groups";
    public static final String USER_ID = "user_id";
    public static final String USER_EMAIL = "email";
    public static final String USER_LOGIN = "bid";
    public static final String USER_INNER_NUMBER = "userDataId";
    public static final String ISSUER = "usb";
    public static final String PURCHASES = "prchs";
    public static final String LANG = "lang";

    public TokenProviderClient(WebContext context) {
        super(context);
    }

    public UserTokenDTO validateToken(TokenData token) {
        UserTokenDTO dto = new UserTokenDTO();
        dto.setToken(token.getToken());
        dto.setScope(token.getScope());
        return validateToken(dto);
    }

    @SneakyThrows
    public UserTokenDTO validateToken(UserTokenDTO dto) {
        try {
            Jws<Claims> claimsJws = Jwts.parser().setSigningKey(getSecretKey(dto.getScope()).getBytes()).build().parseSignedClaims(dto.getToken().replace("Bearer ", ""));
            Date due = claimsJws.getBody().getExpiration();

            if (due.getTime() < new Date().getTime()) {
                throw new TokenExpiredException(dto.getToken());
            }

            dto.setDue(due);
            dto.setRoleListFromGD((List<String>) claimsJws.getBody().get(AUTHORITIES_KEY));

            return dto;
        } catch (SecurityException | MalformedJwtException e) {
            log.info("Invalid JWT signature.");
            log.trace("Invalid JWT signature trace: {0}", e);
        } catch (ExpiredJwtException e) {
            log.info("Expired JWT token.");
            log.trace("Expired JWT token trace: {0}", e);
        } catch (UnsupportedJwtException e) {
            log.info("Unsupported JWT token.");
            log.trace("Unsupported JWT token trace: {0}", e);
        } catch (IllegalArgumentException e) {
            log.info("JWT token compact of handler are invalid.");
            log.trace("JWT token compact of handler are invalid trace: {0}", e);
        }
        return null;
    }

    /**
     * Creates {@link UserTokenDTO} object by token
     * @param tokenData to parse
     * @return Filled object with user Id, and int's roles
     */
    public UserTokenDTO getAuthentication(TokenData tokenData) {
        var token = tokenData.getToken();
        if (token == null || token.trim().isEmpty() || token.equals("null")){
            return null;
        }
        Claims claims = Jwts.parser()
                .setSigningKey(getSecretKey(tokenData.getScope()).getBytes()).build()
                .parseSignedClaims(token).getBody();


        UserTokenDTO dto = new UserTokenDTO();
        dto.setDue(claims.getExpiration());
        dto.setRoleListFromGD((List<String>) claims.get(AUTHORITIES_KEY));
        dto.setToken(token);
        dto.setScope(dto.getScope());
        UserData userData = new UserData();
        userData.setLogin(claims.getSubject());
        userData.setId(Long.valueOf(String.valueOf(claims.get(USER_ID))));
        userData.setEmail(String.valueOf(claims.get(USER_EMAIL)));
        userData.setLogin(String.valueOf(claims.get(USER_LOGIN)));
        userData.setPurchases((List<String>) claims.get(PURCHASES));
        dto.setUserData(userData);
        return dto;

    }

    public String getSecretKey(String scope) {
        try {
            var res =  getContext().getEnv("dennis.systems.security.secret" + "_" + scope);
            if (res == null){
                throw new RuntimeException();
            }
            return String.valueOf(res);
        }catch (Exception e){
            throw new ScopeException( "global.scope.not_supported");
        }
    }
}
