package systems.dennis.auth.service;

import lombok.SneakyThrows;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.utils.SecurityUtils;
import systems.dennis.auth.exception.RoleAlreadyAssignedException;
import systems.dennis.auth.exception.RoleAlreadyUnAssignedException;
import systems.dennis.auth.form.RoleToUserForm;
import systems.dennis.auth.repository.RolesToUserRepo;
import systems.dennis.auth.repository.UserRoleRepo;
import systems.dennis.auth.role_validator.entity.RolesToUser;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemForAddContainsIdException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.postgres.service.PaginationService;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.scopes.repository.ScopeRepo;
import systems.dennis.shared.scopes.service.ScopeService;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
@DataRetrieverDescription(model = RolesToUser.class, form = RoleToUserForm.class, repo = RolesToUserRepo.class)
public class RoleToUserService extends PaginationService<RolesToUser> {

    public RoleToUserService(WebContext context) {
        super(context);
    }


    @Override
    public RolesToUserRepo getRepository() {
        return super.getRepository();
    }

    public List<UserRole> findByOwner(LoginPassword password, ScopeModel scope) {
        AbstractDataFilter<RolesToUser> specification = getFilterImpl().setJoinOn("role").setComplex(true).eq("scope", scope)
                .and(getFilterImpl().eq("password", password));
        return getRepository().filteredData(specification).stream().map(RolesToUser::getRole).collect(Collectors.toList());
    }

    @SneakyThrows
    public RolesToUser applyRole(UserRole role, LoginPassword lp) {
        RolesToUser rolesToUser = new RolesToUser();
        rolesToUser.setRole(role);
        rolesToUser.setPassword(lp);
        rolesToUser.setUserDataId(1L);

        if (existForUser(role, lp)) {
            throw new RoleAlreadyAssignedException(role);
        }

        return getRepository().save(rolesToUser);
    }

    private boolean existForUser(UserRole role, LoginPassword lp) {
        AbstractDataFilter<RolesToUser> specification = getFilterImpl().eq("password", lp)
                .and(getFilterImpl().eq("role", role));
        return getRepository().filteredCount(specification) > 0;
    }

    @SneakyThrows
    @Override
    public RolesToUser preAdd(RolesToUser object) throws ItemForAddContainsIdException {
        if (getRepository().existsByRoleAndPassword(object.getRole(), object.getPassword())) {
            throw new RoleAlreadyAssignedException(object.getRole());
        }
        return super.preAdd(object);
    }

    @Override
    public RolesToUser preDelete(RolesToUser object) {
        if (getRepository().existsByRoleAndPassword(object.getRole(), object.getPassword())) {
            throw new RoleAlreadyAssignedException(object.getRole());
        }
        return super.preDelete(object);
    }

    public boolean removeRole(UserRole role, LoginPassword lp) {
        RolesToUser rolesToUser = new RolesToUser();
        rolesToUser.setRole(role);
        rolesToUser.setPassword(lp);
        rolesToUser.setUserDataId(1L);
        if (!getRepository().existsByRoleAndPassword(role, lp)) {
            throw new RoleAlreadyUnAssignedException(role);
        }

        var roleToRemove = findByRoleAndUser(role, lp);

        getRepository().delete(roleToRemove.get());
        return true;
    }

    public Optional<RolesToUser> findByRoleAndUser(UserRole roleId, LoginPassword password) {
        return getRepository().findByPasswordAndRole(password, roleId);
    }

    @Override
    public AbstractDataFilter<RolesToUser> getAdditionalSpecification() {
        String scopeName = getBean(SecurityUtils.class).getToken().getScope();
        ScopeModel scope = getBean(ScopeRepo.class).filteredFirst(getBean(ScopeService.class).getFilterImpl().eq("name", scopeName))
                .orElseThrow(() -> new ItemNotFoundException(scopeName));
        List<UserRole> roles = getBean(UserRoleRepo.class)
                .filteredData(getBean(RoleServiceImpl.class).getFilterImpl().eq("scope", scope)).toList();

        return super.getAdditionalSpecification().and(getFilterImpl().iN("role", roles));
    }
}
