package systems.dennis.auth.service;


import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.utils.SecurityUtils;
import systems.dennis.auth.form.RoleForm;
import systems.dennis.auth.repository.RolesToUserRepo;
import systems.dennis.auth.repository.UserRoleRepo;
import systems.dennis.auth.role_validator.entity.RolesToUser;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.postgres.service.PaginationService;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.scopes.repository.ScopeRepo;
import systems.dennis.shared.scopes.service.ScopeService;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
@DataRetrieverDescription(model = UserRole.class, form = RoleForm.class, repo = UserRoleRepo.class)
public class RoleServiceImpl extends PaginationService<UserRole>{

    private final RoleToUserService roleToUserService;

    public RoleServiceImpl(WebContext context, RoleToUserService roleToUserService) {
        super(context);
        this.roleToUserService = roleToUserService;
    }


    public boolean existsByName(String value) {
        return getRepository().existsByRole(value);
    }


    public UserRole findFirstByName(String value, Long id) {
        return getRepository().findFirstByRoleAndId(value, id).orElse(null);
    }


    public void applyRolesToUser(LoginPassword lp, ScopeModel scope) {

        var rolesAutoApply = getRepository().filteredData((AbstractDataFilter<UserRole>) getFilterImpl().eq("scope", scope).and(getFilterImpl().eq("addToEveryUser", Boolean.TRUE)));
        for (UserRole role : rolesAutoApply) {
            roleToUserService.applyRole(role, lp);
        }
    }


    @Override
    public UserRoleRepo getRepository() {
        return super.getRepository();
    }

    public List<UserRole> getMyRoles(ScopeModel scope) {

        SecurityUtils utils = getUtils();

        var login = getBean(LoginPasswordService.class).findUserByLogin(utils.get().getUserData().getLogin()).orElse(null);

        AbstractDataFilter specification = getFilterImpl().setComplex(true).setJoinOn("role").eq("scope", scope).and(getFilterImpl().eq("password", login));

        var rolesToUsers = getBean(RoleToUserService.class).getRepository().filteredData(specification);

        var res = new ArrayList<UserRole>();

        rolesToUsers.forEach(x -> res.add( ((RolesToUser)x).getRole()));

        return res;
    }

    public boolean deAssignRole(Long roleId, Long passwordId) {
        return getBean(RoleToUserService.class).removeRole(getRepository().findById(roleId).orElseThrow(() -> ItemNotFoundException.fromId("Role not found -> " + roleId)),
                getBean(LoginPasswordService.class).findById(passwordId).orElseThrow(() -> ItemNotFoundException.fromId("Login Password not found: " + passwordId)));
    }


    public boolean deAssignRole(Long roleId, String login) {


        return getBean(RoleToUserService.class).removeRole(getRepository().findById(roleId).orElseThrow(() -> ItemNotFoundException.fromId("Role not found -> " + roleId)),
                getBean(LoginPasswordService.class).findUserByLogin(login).orElseThrow(() -> ItemNotFoundException.fromId("Login Password not found: " + login)));
    }

    public List<UserData> usersByRole(Long roleId) {
        AbstractDataFilter<RolesToUser> filter = getBean(RoleToUserService.class).getFilterImpl().eq("role", getBean(UserRoleRepo.class).findById(roleId).orElseThrow(() -> ItemNotFoundException.fromId(roleId)));
        var rolesToUser = getBean(RolesToUserRepo.class).filteredData(filter).toList();

        var userDataService = getBean(ProfilePageService.class);

        return rolesToUser.stream().map(x -> userDataService.findByLogin(x.getPassword().getLogin()).orElse(null)).filter(Objects::nonNull).collect(Collectors.toList());

    }

    public Boolean assignRole(Long roleId, String login) {
        var lp=  getBean(LoginPasswordService.class).findUserByLogin(login).orElseThrow(() -> ItemNotFoundException.fromId("Login Password not found: " + login));
        var role = getRepository().findById(roleId).orElseThrow(()-> ItemNotFoundException.fromId( login));
        roleToUserService.applyRole(role, lp);
        return true;
    }

    public UserRole generateAdminRole(ScopeModel scope) {
        UserRole userRole = new UserRole();
        userRole.setRole(UserRole.ROLE_ADMIN);
        userRole.setScope(scope);
        userRole.setAddToEveryUser(false);
        return getRepository().save(userRole);
    }

    public boolean isRelationExist(String role, ScopeModel scope) {
        AbstractDataFilter<UserRole> specification = getFilterImpl().eq("role", role)
                .and(getFilterImpl().eq("scope", scope));
        return count(specification) > 0;
    }

    @Override
    public AbstractDataFilter<UserRole> getAdditionalSpecification() {
        String scopeName = getBean(SecurityUtils.class).getToken().getScope();
        ScopeModel scope = getBean(ScopeRepo.class).filteredFirst(getBean(ScopeService.class).getFilterImpl().eq("name", scopeName))
                .orElseThrow(() -> new ItemNotFoundException(scopeName));
        AbstractDataFilter<UserRole> filter = getFilterImpl().eq("scope", scope);

        return super.getAdditionalSpecification().and(filter);
    }
}
