package systems.dennis.auth.service;

import org.springframework.stereotype.Service;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.entity.UserDataForm;
import systems.dennis.auth.exception.VerificationException;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.DeleteStrategy;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemForAddContainsIdException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.exceptions.ItemNotUserException;
import systems.dennis.shared.exceptions.UnmodifiedItemSaveAttemptException;
import systems.dennis.shared.postgres.service.PaginationService;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.scopes.model.ScopeModel;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static systems.dennis.shared.annotations.DeleteStrategy.DELETE_STRATEGY_PROPERTY;


@Service
@DataRetrieverDescription(form = UserDataForm.class, model = UserData.class, repo = UserDataRepository.class)
@DeleteStrategy(value =  DELETE_STRATEGY_PROPERTY)
public class ProfilePageService extends PaginationService<UserData> {

    public ProfilePageService(WebContext context) {
        super(context);
    }

    @Override
    public UserData preAdd(UserData object) throws ItemForAddContainsIdException {
        object.setLastUpdate(new Date());
        return super.preAdd(object);
    }

    @Override
    public UserData preEdit(UserData object, UserData original) throws UnmodifiedItemSaveAttemptException, ItemNotFoundException {
        object.setLastUpdate(new Date());
        return super.preEdit(object, original);
    }

    public Optional<UserData> findByLogin(String login) {
        return getRepository().findByLogin(login);
    }

    public Optional<UserData> findByEmail(String email) {
        AbstractDataFilter<UserData> filter = getFilterImpl().eq("email", email);
        return getRepository().filteredFirst(filter);
    }

    @Override
    public  UserDataRepository getRepository() {
        return super.getRepository();
    }

    public UserData savePhoto(Long profileId, String url) {
        var item = getRepository().findById(profileId).orElseThrow(()-> ItemNotFoundException.fromId(profileId));

        getUtils().isMy(item);

        item.setImagePath(url);

        return this.save(item);

    }



    public String getCurrentUserImage(){
        //

        UserData res = findById(getCurrentUser(), ItemNotFoundException.fromId(getCurrentUser()));
        return res.getImagePath() == null ? "/images/profile_photo.png" : res.getImagePath();
    }

    public List<UserRole> myRoles(ScopeModel scope) {

        return getBean(RoleServiceImpl.class).getMyRoles(scope);
    }

    @Override
    public void checkMy(UserData object) {
        if (!getCurrentUser().equals(object.getId())) {
            throw new ItemNotUserException();
        }
    }

    public void checkVerifiedUser(UserData user, ScopeModel scope) {
        if (Objects.nonNull(scope.getScopeRule()) && scope.getScopeRule().getVerificationRequired()) {
            if (!user.getVerified()) {
                throw new VerificationException("global.exceptions.user_not_verified");
            }
        }
    }
}
