package systems.dennis.auth.service;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.utils.SecurityUtils;
import systems.dennis.auth.exception.InvitationException;
import systems.dennis.auth.form.InvitationForm;
import systems.dennis.auth.mail.MailSender;
import systems.dennis.auth.model.InvitationModel;
import systems.dennis.auth.repository.InvitationRepo;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.DeleteStrategy;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemForAddContainsIdException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.service.AbstractPaginationService;
import systems.dennis.shared.utils.TSDate;

import java.util.*;

import static systems.dennis.shared.annotations.DeleteStrategy.DELETE_STRATEGY_PROPERTY;

@Service
@DataRetrieverDescription(model = InvitationModel.class, form = InvitationForm.class, repo = InvitationRepo.class)
@DeleteStrategy(value =  DELETE_STRATEGY_PROPERTY)
public class InvitationService extends AbstractPaginationService<InvitationModel, Long> {

    public InvitationService(WebContext holder) {
        super(holder);
    }

    @Override
    public InvitationModel preAdd(InvitationModel object) throws ItemForAddContainsIdException {
        UserData inviter = getBean(SecurityUtils.class).get().getUserData();
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(getContext().getRequest(), inviter.getId(), false);

        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.MINUTE, getInvitationDuration());

        object.setInviter(inviter);
        object.setAccepted(false);
        object.setScope(scope);
        object.setDate(now);
        object.setDateDue(calendar.getTime());

        return super.preAdd(object);
    }

    @Override
    public InvitationModel afterAdd(InvitationModel object) {
        sendInvitationEmail(object);

        return super.afterAdd(object);
    }

    public Boolean accept(InvitationModel invitation) {
            Boolean accepted = invitation.getAccepted();

            if (accepted) {
                throw new InvitationException("global.exceptions.invitation_already_accept");
            }

            Date currentDate = new Date();
            if (invitation.getDateDue().toInstant().isBefore(currentDate.toInstant())) {
                throw new InvitationException("global.exceptions.invitation_expired");
            }

            invitation.setAccepted(true);
            save(invitation);

            return true;
    }

    public List<InvitationModel> getMyAcceptInvitation(Long from, Integer limit, Integer page) {
        UserData inviter = getBean(SecurityUtils.class).get().getUserData();

        //todo use search instead
        return getRepository().filteredData(getFilterImpl().eq("inviter", inviter).and(getFilterImpl().eq("accepted", true)), Pageable.ofSize(limit)).getContent();
    }

    public void addUserToScope(InvitationModel invitation) {
        UserData user = getBean(ProfilePageService.class).findByLogin(invitation.getEmail())
                .orElseThrow(() -> ItemNotFoundException.fromId(invitation.getEmail()));
        ScopeModel scope = invitation.getScope();
        if (!getBean(UserInScopeService.class).isRelationExist(user, scope)) {
            getBean(UserInScopeService.class).generateAndSave(user, scope);
        }
    }

    private void sendInvitationEmail(InvitationModel object) {
        String inviteUrl = generateInviteUrl(object);
        String content = prepareEmailContent(object, inviteUrl);
        String lang = object.getInviter().getPreferredLanguage();
        String title = getContext().getMessageTranslation("email.invitation.title", lang);
        getBean(MailSender.class).sendMail(Collections.singletonList(object.getEmail()), content, title);
    }

    private String prepareEmailContent(InvitationModel invitation, String inviteUrl) {
        Map<String, String> templateModel = new HashMap<>();
        templateModel.put("inviter", invitation.getInviter().getLogin());
        templateModel.put("invitationLink", inviteUrl);
        String path = invitation.getInviter().getPreferredLanguage() + "/invitation.html";
        return getBean(MailSender.class).processHtmlTemplate(path, templateModel);
    }

    private String generateInviteUrl(InvitationModel object) {
        ScopeModel scope = object.getScope();
        String baseUrl = scope.getUrl();

        return baseUrl + getInvitationInfoPath() + "?id=" + object.getId();
    }

    private int getInvitationDuration() {
        return getContext().getEnv("invitation.duration", 43200);
    }

    private String getInvitationInfoPath() {
        return getContext().getEnv("app.invitation.info.path", "/invitation/info");
    }
}
