package systems.dennis.auth.service;

import jakarta.servlet.http.HttpServletRequest;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.config.AuthorizationDelegator;
import systems.dennis.auth.exception.ScopeException;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.DeleteStrategy;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemForAddContainsIdException;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.scopes.exception.ScopeRuleException;
import systems.dennis.shared.scopes.form.ScopeForm;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.scopes.repository.ScopeRepo;
import systems.dennis.shared.scopes.service.ScopeService;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

import static systems.dennis.shared.annotations.DeleteStrategy.DELETE_STRATEGY_PROPERTY;

@Service
@DataRetrieverDescription(model = ScopeModel.class, form = ScopeForm.class, repo = ScopeRepo.class)
@DeleteStrategy(value = DELETE_STRATEGY_PROPERTY)
@Primary
public class AuthScopeService extends ScopeService {
    private static final int KEY_LENGTH = 100;
    private final ScopeRepo scopeRepo;

    public AuthScopeService(WebContext holder,
                            ScopeRepo scopeRepo) {
        super(holder);
        this.scopeRepo = scopeRepo;
    }

    @Override
    public ScopeModel preAdd(ScopeModel object) throws ItemForAddContainsIdException {
        String secretKey = getBean(PasswordService.class).generateRandomPassword(KEY_LENGTH);
        object.setSecretKey(secretKey);
        return super.preAdd(object);
    }

    @Override
    public ScopeModel afterAdd(ScopeModel object) {
        addCreatorToScope(object);
        assignAdminRole(object);
        return super.afterAdd(object);
    }

    public ScopeModel getScopeFromRequest(HttpServletRequest req, Long userId, boolean fullAccess) {
        if (req == null) {
            throw new ScopeException("global.exceptions.failed_determine_scope");
        }

        String scopeId = req.getHeader(AuthorizationDelegator.AUTH_SCOPE_HEADER);

        if (scopeId == null) {
            throw new ScopeException("global.exceptions.failed_determine_scope");
        }

        return findByName(scopeId, userId, fullAccess);
    }

    private void addCreatorToScope(ScopeModel scope) {
        UserData userData = getBean(ProfilePageService.class).findByIdOrThrow(scope.getUserDataId());
        getBean(UserInScopeService.class).generateAndSave(userData, scope);
    }

    private void assignAdminRole(ScopeModel scope) {
        UserRole roleAdmin = getBean(RoleServiceImpl.class).generateAdminRole(scope);
        UserData userData = getBean(ProfilePageService.class).findByIdOrThrow(scope.getUserDataId());
        LoginPassword lp = getBean(LoginPasswordService.class).findUserByLogin(userData.getEmail()).orElseThrow();
        getBean(RoleToUserService.class).applyRole(roleAdmin, lp);
    }

    @Override
    public AbstractDataFilter<ScopeModel> addSelfSpecWithUserScopes(Long userData) {
        AbstractDataFilter<ScopeModel> selfSpec = getSelfCreatedItems(userData);

        return addScopeSpecification(selfSpec, userData);
    }

    @Override
    public AbstractDataFilter<ScopeModel> getAdditionalSpecification() {
        AbstractDataFilter<ScopeModel> additionalSpec = super.getAdditionalSpecification();
        AbstractDataFilter<ScopeModel> selfSpec = getSelfCreatedItems(getCurrentUser());

        return addScopeSpecification(selfSpec, getCurrentUser()).and(additionalSpec);
    }

    public ScopeModel generateAndSaveUserScope(UserData userData) {
        ScopeModel selfScope = new ScopeModel();
        selfScope.setName(UUID.randomUUID().toString());
        selfScope.setUserDataId(userData.getId());
        selfScope.setDate(new Date());
        selfScope.setSecretKey(getBean(PasswordService.class).generateRandomPassword(KEY_LENGTH));
        selfScope = scopeRepo.save(selfScope);

        addCreatorToScope(selfScope);
        assignAdminRole(selfScope);
        return selfScope;
    }

    public void checkRegistrationAllowed(ScopeModel scope) {
        if (Objects.nonNull(scope.getScopeRule()) && !scope.getScopeRule().getRegistrationAllowed()) {
            throw new ScopeRuleException("global.exceptions.registration_not_allowed");
        }
    }

    private AbstractDataFilter<ScopeModel> addScopeSpecification(AbstractDataFilter<ScopeModel> selfSpec, Long userData) {
        List<Long> scopeIds = getBean(UserInScopeService.class).getByUser(userData)
                .stream()
                .filter(x -> x.getScope() != null)
                .map((x) -> x.getScope().getId())
                .collect(Collectors.toList());

        AbstractDataFilter<ScopeModel> userInScopeSpec = getFilterImpl().iN("id", scopeIds);

        return (selfSpec != null) ? selfSpec.or(userInScopeSpec) : userInScopeSpec;
    }
}
