package systems.dennis.auth.delegations.simple;

import lombok.extern.slf4j.Slf4j;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.entity.ActiveToken;
import systems.dennis.auth.entity.LoginHistory;
import systems.dennis.auth.repository.LoginHistoryRepository;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.role_validator.TokenProvider;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.LoginPasswordService;
import systems.dennis.auth.service.UserInScopeService;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.scopes.model.ScopeModel;

import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
public class SimpleTokenAuthorization implements AuthorizationAttemptProcessor {
    @Override
    public UserTokenDTO authorize(LoginPassword password, WebContext.LocalWebContext context, ScopeModel scope) {

        var passwordService = context.getBean(PasswordService.class);
        var loginPasswordService = context.getBean(LoginPasswordService.class);
        var tokenProvider = context.getBean(TokenProvider.class);

        LoginPassword res = null;

        password.setPassword(passwordService.toPassword(password.getPassword()));

        res = loginPasswordService.findOrThrow(password, true);
        if (res == null) throw new AuthorizationFailedException(password.getLogin());

        UserData userData = createUser(res, context);

        if (!context.getBean(UserInScopeService.class).isRelationExist(userData, scope)) {
            throw new AuthorizationFailedException(password.getLogin());
        }

        UserTokenDTO dto = new UserTokenDTO();
        dto.setUserData(userData);
        log.debug("Authorization ->  , creating token: {}", password.getLogin());
        var roles = passwordService.getRoles(res, scope);
        var token = tokenProvider.createToken(dto, ActiveToken.DEFAULT_TOKEN_TYPE, roles, scope);
        dto.setToken(token.getToken());
        dto.setDue(token.getDue());
        dto.setRoles(roles.stream().map(UserRole::getRole).collect(Collectors.toList()));

        saveLoginAttempt(dto, context);
        return dto;


    }

    @Override
    public UserData createUser(LoginPassword loginPassword, WebContext.LocalWebContext context) {

        final UserData temp = new UserData();
        temp.setLogin(loginPassword.getLogin());
        Optional<UserData> user = context.getBean(UserDataRepository.class).findByLogin(temp.getLogin());
        return user.orElseGet(() -> createUser(temp));


    }


    private UserData createUser(UserData temp) {

        UserData newUserData = new UserData();
        newUserData.setLogin(temp.getEmail());
        newUserData.setEmail(temp.getEmail());
        return newUserData;

    }

    @Override
    public void saveLoginAttempt(UserTokenDTO user, WebContext.LocalWebContext context) {
        LoginHistory history = new LoginHistory();
        log.debug("TRacing Login history started");
        history.setUserDataId(user.getUserData().getId());
        history.setToken(user.getToken());
        history.setAuthorizationType(DefaultAuthorizationDelegator.AUTH_TYPE_DEFAULT);
        history.setLogin(user.getUserData().getLogin());
        context.getBean(LoginHistoryRepository.class).save(history);

    }
}
