package systems.dennis.auth.client.utils;


import de.taimos.totp.TOTP;
import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base32;
import org.apache.commons.codec.binary.Hex;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.required.TokenProviderClient;
import systems.dennis.auth.exception.SubscriptionNotExistsException;
import systems.dennis.auth.repository.LoginPasswordRepo;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.responses.Auth2FactorEnabled;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.LoginPasswordService;
import systems.dennis.shared.annotations.security.ISecurityUtils;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.entity.TokenData;
import systems.dennis.shared.exceptions.AccessDeniedException;
import systems.dennis.shared.exceptions.AuthorizationNotFoundException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.utils.ApplicationContext;
import systems.dennis.shared.utils.bean_copier.BeanCopier;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static systems.dennis.auth.role_validator.entity.UserRole.ROLE_ADMIN;


@Slf4j
@Service
public class SecurityUtils implements ISecurityUtils<Long> {


    @Autowired
    private TokenProviderClient provider;

    @Autowired
    private AuthenticationService service;
    @Autowired
    HttpServletRequest request;


    private SecurityUtils() {
    }

    /**
     * Masks a string withing half of its symbols
     *
     * @param token a String to mask
     * @return a masked string
     */
    public static String mask(String token) {
        if (token == null || token.length() < 2) {
            return "***********";
        }


        String stars = "*".repeat(token.length() / 2);
        return stars + token.substring(stars.length());
    }


    public boolean isAdmin() {
        try {
            var auth = get().getRoleList();
            return auth.contains(ROLE_ADMIN);

        } catch (Exception e){
            return false;
        }

    }

    @Override
    public boolean hasRole(String role) {
        return service.hasRole(role);
    }

    @Override
    public String getUserLanguage() {
        return get().getUserData().getPreferredLanguage();
    }

    @SneakyThrows
    public void removeLocalAuthorization() {
        service.logout();
    }

    public UserTokenDTO get() throws AuthorizationNotFoundException {
        return tokenFromHeader();
    }

    @SneakyThrows
    public Long getUserDataId() {
        return get().getUserData().getId();
    }



    public void checkSubscription(List<String> possibleSubscriptions) {
        var purchases = get().getUserData().getPurchases();

        if (purchases == null || purchases.isEmpty()) {
            throw new SubscriptionNotExistsException();
        }

        for (String purchase : purchases) {
            if (possibleSubscriptions.contains(purchase)) {
                return;
            }
        }

        throw new SubscriptionNotExistsException();
    }

    @SneakyThrows
    public UserTokenDTO tokenFromHeader() throws AuthorizationNotFoundException {

        return provider.getAuthentication(getTokenFromRequest());

    }

    public String generateSecretKey() {
        SecureRandom random = new SecureRandom();
        byte[] bytes = new byte[20];
        random.nextBytes(bytes);
        Base32 base32 = new Base32();
        return base32.encodeToString(bytes);
    }

    @SneakyThrows
    public Auth2FactorEnabled get2factorBarCodeForUser(LoginPasswordService loginPasswordService) {
        var login = getLoginData(loginPasswordService);
        String secretKey = Optional.ofNullable(login.getTwoFactorCode()).orElseGet(this::generateSecretKey);
        if (login.getTwoFactorCode() == null) {
            login.setTwoFactorCode(secretKey);
            loginPasswordService.save(login);
        }
        String email = login.getLogin();
        String companyName = "dennis.systems";
        String code = getGoogleAuthenticatorBarCode(secretKey, email, companyName);

        Auth2FactorEnabled res = new Auth2FactorEnabled();
        res.setCode(code);
        res.setEnabled(login.getTwoFactor() == null ? false : login.getTwoFactor());

        return res;
    }

    public LoginPassword getLoginData(LoginPasswordService service) {
        return service.findUserByLogin(get().getUserData().getLogin()).orElseThrow(() -> new AuthorizationNotFoundException(""));
    }

    @SneakyThrows
    public String getGoogleAuthenticatorBarCode(String secretKey, String account, String issuer) {

        return "otpauth://totp/"
                + URLEncoder.encode(issuer + ":" + account, StandardCharsets.UTF_8).replace("+", "%20")
                + "?secret=" + URLEncoder.encode(secretKey, StandardCharsets.UTF_8).replace("+", "%20")
                + "&issuer=" + URLEncoder.encode(issuer, StandardCharsets.UTF_8).replace("+", "%20");
    }

    @SneakyThrows
    public String getTOTPCode(LoginPasswordService service, String login) {
        var user = service.findUserByLogin(login).orElseThrow((() -> new AuthorizationNotFoundException(" No such user: " + login)));

        if (user.getTwoFactor() == null || !user.getTwoFactor()) {
            return null;
        }

        Base32 base32 = new Base32();
        byte[] bytes = base32.decode(user.getTwoFactorCode());
        String hexKey = Hex.encodeHexString(bytes);
        return TOTP.getOTP(hexKey);
    }

    @SneakyThrows
    public String getTOTPCode(LoginPassword lp) {
        Base32 base32 = new Base32();
        byte[] bytes = base32.decode(lp.getTwoFactorCode());
        String hexKey = Hex.encodeHexString(bytes);
        return TOTP.getOTP(hexKey);
    }

    /**
     * Returns a token from request
     *
     * @return string value or null
     */
    public TokenData getTokenFromRequest() throws AuthorizationNotFoundException {

        return service.getToken(request);
    }


    /**
     * @return
     * @throws AuthorizationNotFoundException
     */
    public TokenData getTokenOrThrow() throws AuthorizationNotFoundException, AuthorizationNotFoundException {
        var authentication = get();
        TokenData token = null;
        if (authentication == null) {
            token = getTokenFromRequest();
        }


        if (token == null) {
            if (get() == null) {
                throw new AuthorizationNotFoundException("NO TOKEN PROVIDED");
            }
            var tokenData = get();
            token = new TokenData(get().getScope(), get().getToken());

            if (token == null) {
                throw new AuthorizationNotFoundException(" NO TOKEN PROVIDED");
            }
        }
        return token;
    }

    public UserData userOrThrow() throws AuthorizationNotFoundException {
        return get().getUserData();
    }

    public List<String> roles() throws AuthorizationNotFoundException {
        return get().getRoleList();
    }

    public boolean roleExists(String... roles) throws AuthorizationNotFoundException {
        for (String role : roles()) {
            for (String r : roles) {
                if (r.equalsIgnoreCase(role)) return true;
            }
        }
        return false;
    }

    public void roleExistsOrThrow(String... roles) throws AuthorizationNotFoundException {
        if (!roleExists(roles)) {
            throw new AccessDeniedException("User has not required roles " + Arrays.toString(roles));
        }
    }

    @SneakyThrows
    public boolean anyRole() {
        return !roles().isEmpty();
    }

    public void isMe(Long id) {

        var isMe = getUserDataId().equals(id) || isAdmin();

        if (!isMe) {
            throw new AccessDeniedException("You cannot change this object");
        }
    }

    public TokenData getToken() {
        return getTokenFromRequest();
    }

    public void isMy(Object item) {
        if (isAdmin()) {
            //admin can do everything he wants to
            return;
        }


        try {
            var value = BeanCopier.readValue(item, ApplicationContext.CREATED_FIELDS_MAP.get(item.getClass()));
            if (!Objects.equals(value, get().getUserData())){
                throw new AccessDeniedException("You have no access to this object");
            }
        } catch (AuthorizationNotFoundException e){
            throw new AccessDeniedException("You have no access to this object");
        } catch (Exception e){
            return;
        }

    }

    @SneakyThrows
    public void isMePswrd(Long id, WebContext.LocalWebContext context) {
        if (id == null) return;

        var loginPasswordRepo = context.getBean(LoginPasswordRepo.class);
        var userDataRepository = context.getBean(UserDataRepository.class);
        var item = loginPasswordRepo.findById(id).orElseThrow(() -> ItemNotFoundException.fromId(id));
        var userData = userDataRepository.findByLogin(item.getLogin()).orElseThrow(() -> ItemNotFoundException.fromId(id));

        isMe(userData.getId());
    }
}
