package systems.dennis.auth.service;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.utils.SecurityUtils;
import systems.dennis.auth.form.LoginPasswordForm;
import systems.dennis.auth.form.RegistrationForm;
import systems.dennis.auth.repository.LoginPasswordRepo;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.repository.VirtualLoginPasswordRepo;
import systems.dennis.auth.role_validator.entity.RolesToUser;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.DeleteStrategy;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.items.magic.MagicRequest;
import systems.dennis.shared.exceptions.*;
import systems.dennis.shared.postgres.repository.PaginationRepository;
import systems.dennis.shared.postgres.service.PaginationService;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.utils.bean_copier.BeanCopier;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static systems.dennis.shared.annotations.DeleteStrategy.DELETE_STRATEGY_PROPERTY;

@Service

@DataRetrieverDescription(model = LoginPassword.class, form = LoginPasswordForm.class, repo = LoginPasswordRepo.class)
@DeleteStrategy(value = DELETE_STRATEGY_PROPERTY)
public class LoginPasswordService extends PaginationService<LoginPassword> {
    private WebContext.LocalWebContext context;
    private final LoginPasswordRepo repo;
    private final UserDataRepository userRepository;
    private final RoleServiceImpl roleService;
    private final PasswordService passwordService;
    private final RoleToUserService roleToUserService;

    public LoginPasswordService(WebContext context, LoginPasswordRepo repo, UserDataRepository userDataRepository, RoleServiceImpl roleService, PasswordService passwordService, RoleToUserService roleToUserService) {
        super(context);
        this.context = WebContext.LocalWebContext.of("login.password.service", context);
        this.repo = repo;
        this.userRepository = userDataRepository;
        this.roleService = roleService;
        this.passwordService = passwordService;
        this.roleToUserService = roleToUserService;
    }

    @Override
    public PaginationRepository<LoginPassword> getRepository() {
        return repo;
    }

    public UserData saveUserData(RegistrationForm form) {
        UserData data = new UserData();
        data.setLogin(form.getEmail());
        data.setEmail(form.getEmail());
        data.setName(form.getName());
        return userRepository.save(data);
    }

    public LoginPassword formToPojo(RegistrationForm form) {
        return LoginPassword.from(form);
    }

    @Override
    public LoginPassword preAdd(LoginPassword object) throws ItemForAddContainsIdException {
        object.setPassword(passwordService.toPassword(object.getPassword()));
        return object;
    }

    public void saveUserRoles(LoginPassword lp, ScopeModel scope) {
        roleService.applyRolesToUser(lp, scope);
    }

    public List<String> findUsersByLogin(String text) {

        var res = repo.filteredData((AbstractDataFilter<LoginPassword>) getFilterImpl().setInsensitive(true).contains("login", text), PageRequest.of(0, 50));

        var logins = new ArrayList<String>();

        res.forEach(x -> logins.add(x.getLogin()));

        return logins;
    }

    public Optional<LoginPassword> findUserByLogin(String login) {
        return repo.filteredOne((AbstractDataFilter<LoginPassword>) getFilterImpl().eq("login", login));
    }

    public Optional<RolesToUser> findUserAssigment(String user, Long roleId) throws ItemNotFoundException {
        var res = findUserByLogin(user).orElseThrow(() -> ItemNotFoundException.fromId(user));
        var role = roleService.findById(roleId).orElseThrow(() -> ItemNotFoundException.fromId(roleId));
        return roleToUserService.findByRoleAndUser(role, res);
    }

    public RolesToUser assignUser(String userLogin, Long roleId) {
        return roleToUserService.applyRole(roleService.findById(roleId).orElseThrow(), repo.findByLogin(userLogin).orElseThrow());
    }



    @Override
    public  Optional<LoginPassword> findById(Serializable id) {
        var res = super.findById(id);
        if (res.isPresent()){
            var newLP = getBean(BeanCopier.class).copy(res.get(), LoginPassword.class);
            newLP.setPassword("*****");
            return Optional.of( newLP);
        }
        return res;
    }

    @Override

    public Page<LoginPassword> search(MagicRequest request) {
        if (!getUtils().isAdmin()) {
            throw new AccessDeniedException("only.admin.can.search_this");
        }

        var res = super.search(request);

        List<LoginPassword> loginPasswords = new ArrayList<>();

        for (var r : res) {
            LoginPassword newLP = getBean(BeanCopier.class).copy(r, LoginPassword.class);
            newLP.setPassword("********");
            loginPasswords.add(newLP);
        }

        res.getContent().clear();
        res.getContent().addAll(loginPasswords);
        return res;
    }

    public LoginPassword findOrThrow(LoginPassword loginPassword) {
        return findOrThrow(loginPassword, true);
    }

    public Optional<LoginPassword> login(String login, String toPassword, String code) {
        var res = repo.login(login, toPassword);

        if (res.isEmpty() || (res.get().getTwoFactor() == null && res.get().getTwoFactor() == Boolean.FALSE)) {
            // do not check code for the 2factor
            return res;
        }

        var security = context.getBean(SecurityUtils.class).getTOTPCode(this, login);

        if (security == null) {
            return res;
        }

        if (!security.equalsIgnoreCase(code)) {
            throw new AuthorizationNotFoundException("Code was wrong! ");
        }

        return res;
    }

    public LoginPassword findOrThrow(LoginPassword loginPassword, boolean real) {

        if (real) {
            var res = this.login(loginPassword.getLogin(), loginPassword.getPassword(), loginPassword.getTwoFactorCode());
            if (res.isEmpty()) {
                throw new AuthorizationFailedException(loginPassword.getLogin());
            }
            return res.get();
        } else {
            var res = context.getBean(VirtualLoginPasswordRepo.class).login(loginPassword.getLogin(), passwordService.toPassword(loginPassword.getPassword()));            if (res.isEmpty() || !res.get().getIsActive()) {
                throw new AccessDeniedException("User is not found or inactive");
            }

            var user = res.get().getUserDataId();
            var userData = context.getBean(UserDataRepository.class).findById(user).orElseThrow(() -> new AccessDeniedException("The managing owner not found"));

            var login = userData.getLogin();

            return context.getBean(LoginPasswordRepo.class).findByLogin(login).orElseThrow(() -> new AccessDeniedException("User is not present owned by virtual user"));

        }
    }

    public boolean isOnlyOne() {
        return getRepository().count() < 2;
    }
}
