package systems.dennis.auth.delegations.virtual;

import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.config.AuthorizationDelegator;
import systems.dennis.auth.config.AuthorizeResponse;
import systems.dennis.auth.entity.ActiveToken;
import systems.dennis.auth.exception.UserIsBlockedException;
import systems.dennis.auth.exception.VirtualUserAuthorizationException;
import systems.dennis.auth.form.ChangePasswordForm;
import systems.dennis.auth.form.RegistrationForm;
import systems.dennis.auth.repository.LoginPasswordRepo;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.repository.VirtualLoginPasswordRepo;
import systems.dennis.auth.role_validator.TokenProvider;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.AuthScopeService;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.AccessDeniedException;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.scopes.model.ScopeModel;

@Slf4j
public class VirtualUserAuthorizationDelegator implements AuthorizationDelegator {

    public static final String AUTH_TYPE_VIRTUAL = "VIRTUAL";

    @Override
    public AuthorizeResponse authorize(HttpServletRequest request, LoginPassword password, WebContext.LocalWebContext context) {
        ScopeModel scope = context.getBean(AuthScopeService.class).getScopeFromRequest(request, null, true);

        var item = context.getBean(VirtualLoginPasswordRepo.class)
                .login(password.getLogin(), context.getBean(PasswordService.class).toPassword(password.getPassword()))
                .orElseThrow(() -> new AuthorizationFailedException(password.getLogin()));

        if (item.getIsActive() != null && item.getIsActive() == Boolean.FALSE){
            throw new VirtualUserAuthorizationException("User " + password.getLogin() + " is blocked");
        }

        var toAuth = context.getBean(UserDataRepository.class).findById(item.getUserDataId()).orElseThrow(()->new VirtualUserAuthorizationException("User assigned to virtual user does not exist"));

        if (toAuth.getBlocked() == Boolean.TRUE){
            throw new UserIsBlockedException(" User owning this virtual user is blocked");
        }



        var realUserLoginPassword = context.getBean(LoginPasswordRepo.class).findByLogin(toAuth.getLogin()).orElseThrow();

        var auth = new DirectUserLoginAuthorization(AUTH_TYPE_VIRTUAL).authorize(password, context, scope);

        return  AuthorizeResponse.of(auth, false);
    }


    @Override
    public boolean shouldAuthorize(HttpServletRequest request, WebContext.LocalWebContext context) {
        if ( AUTH_TYPE_VIRTUAL.equals(request.getHeader(AUTH_TYPE_HEADER))) {
            log.debug("Header AUTH-TYPE declares to use VirtualUserAuth");
            return true;
        }
        log.debug("Header AUTH-TYPE declares not to use VirtualUserAuth");
        return false;
    }


    @Override
    public boolean blockUser(boolean block, Long userDataId, WebContext.LocalWebContext context) {

        return true;
    }

    @Override
    public boolean logout(String token,  WebContext.LocalWebContext context, ScopeModel scope) {
        context.getBean(TokenProvider.class).removeAuthToken(token, ActiveToken.DEFAULT_VIRTUAL_TYPE, scope);
        return true;
    }

    @Override
    public boolean register(RegistrationForm form, WebContext.LocalWebContext context, ScopeModel scope, Long invitationId) {
        throw new AccessDeniedException("registering virtual users is not possible");
    }

    @SneakyThrows
    @Override
    public void validate(UserTokenDTO dto, WebContext.LocalWebContext context) {
        dto.validate(context);
        if (dto.getUserData().getBlocked() == Boolean.TRUE) {
            throw new UserIsBlockedException("User is blocked");
        }
    }

    @Override
    public boolean changePassword(HttpServletRequest req, WebContext.LocalWebContext context, ChangePasswordForm loginPassword, ScopeModel scope) {


        throw new AccessDeniedException("changing password of virtual users is not possible");
    }

    @Override
    public String forgetPassword(HttpServletRequest req, WebContext.LocalWebContext context, String login) {
        throw new AccessDeniedException("forget password of virtual users is not possible");
    }
}
