package systems.dennis.auth.delegations.virtual;

import lombok.extern.slf4j.Slf4j;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.delegations.simple.AuthorizationAttemptProcessor;
import systems.dennis.auth.entity.ActiveToken;
import systems.dennis.auth.entity.LoginHistory;
import systems.dennis.auth.repository.LoginHistoryRepository;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.role_validator.TokenProvider;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.LoginPasswordService;
import systems.dennis.auth.service.UserInScopeService;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.AccessDeniedException;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.scopes.model.ScopeModel;

import java.util.Optional;
import java.util.stream.Collectors;

import static systems.dennis.auth.delegations.virtual.VirtualUserAuthorizationDelegator.AUTH_TYPE_VIRTUAL;

@Slf4j
public class DirectUserLoginAuthorization implements AuthorizationAttemptProcessor {
    private String type;

    public DirectUserLoginAuthorization(String type){

        this.type = type;
    }
    @Override
    public UserTokenDTO authorize(LoginPassword password, WebContext.LocalWebContext context, ScopeModel scope) {

        var passwordService = context.getBean(PasswordService.class);
        var loginPasswordService = context.getBean(LoginPasswordService.class);
        var tokenProvider = context.getBean(TokenProvider.class);

        LoginPassword res = null;


        try {
            res = loginPasswordService.findOrThrow(password, !AUTH_TYPE_VIRTUAL.equals(type));
            if (res == null) throw new AccessDeniedException(password.getLogin());
        } catch (AccessDeniedException e) {
            //In order not to move to login page, and throw correct exception, we will throw a separate exception which will
            //throw 401 status on error
            throw new AuthorizationFailedException(password.getLogin());
        }

        UserData userData = createUser(res, context);

        if (!context.getBean(UserInScopeService.class).isRelationExist(userData, scope)) {
            throw new AuthorizationFailedException(userData.getLogin());
        }

        UserTokenDTO dto = new UserTokenDTO();
        dto.setUserData(userData);
        log.debug("Authorization ->  , creating token: {}", password.getLogin());
        var roles = passwordService.getRoles(res, scope);
        var token = tokenProvider.createToken(dto, ActiveToken.DEFAULT_VIRTUAL_TYPE, roles, scope);
        dto.setToken(token.getToken());
        dto.setDue(token.getDue());
        dto.setRoles(roles.stream().map(UserRole::getRole).collect(Collectors.toList()));

        saveLoginAttempt(dto, context);
        return dto;


    }


    @Override
    public UserData createUser(LoginPassword loginPassword, WebContext.LocalWebContext context) {

        final UserData temp = new UserData();
        temp.setLogin(loginPassword.getLogin());
        Optional<UserData> user = context.getBean(UserDataRepository.class).findByLogin(temp.getLogin());
        return user.orElseGet(() -> createUser(temp));


    }


    private UserData createUser(UserData temp) {

        UserData newUserData = new UserData();
        newUserData.setLogin(temp.getLogin());
        newUserData.setEmail(temp.getLogin());
        return newUserData;

    }

    @Override
    public void saveLoginAttempt(UserTokenDTO user, WebContext.LocalWebContext context) {
        LoginHistory history = new LoginHistory();
        log.debug("TRacing Login history started");
        history.setUserDataId(user.getUserData().getId());
        history.setToken(user.getToken());
        history.setAuthorizationType(type);
        history.setLogin(user.getUserData().getLogin());
        context.getBean(LoginHistoryRepository.class).save(history);

    }
}
