package systems.dennis.auth.delegations.simple;

import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.utils.AuthenticationService;
import systems.dennis.auth.client.utils.SecurityUtils;
import systems.dennis.auth.config.AuthorizationDelegator;
import systems.dennis.auth.config.AuthorizeResponse;
import systems.dennis.auth.entity.ActiveToken;
import systems.dennis.auth.exception.UserIsBlockedException;
import systems.dennis.auth.form.ChangePasswordForm;
import systems.dennis.auth.form.RegistrationForm;
import systems.dennis.auth.model.InvitationModel;
import systems.dennis.auth.repository.ActiveTokensRepo;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.role_validator.TokenProvider;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.*;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.forms.ValidateForm;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.exceptions.StandardException;
import systems.dennis.shared.exceptions.ValidationFailedException;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.servers.repository.ServerConfigRepo;

import java.rmi.AccessException;
import java.util.Date;
import java.util.Objects;
import java.util.Optional;

@Slf4j
public class DefaultAuthorizationDelegator implements AuthorizationDelegator {



    public static final String AUTH_TYPE_DEFAULT = "DEFAULT";




    @Override
    public AuthorizeResponse authorize(HttpServletRequest request, LoginPassword password, WebContext.LocalWebContext context) {
        UserData user = context.getBean(ProfilePageService.class).findByLogin(password.getLogin()).orElseThrow(() -> ItemNotFoundException.fromId(password.getLogin()));
        ScopeModel scope = context.getBean(AuthScopeService.class).getScopeFromRequest(request, user.getId(), false);
        context.getBean(ProfilePageService.class).checkVerifiedUser(user, scope);

        var res = new SimpleTokenAuthorization().authorize(password, context, scope);

        if(res == null ) {
            log.debug("Authorization failed. We throw authorization exception");
            invokeAuthError( password);
        }

        return AuthorizeResponse.of(res, false);
    }

    private void invokeAuthError(LoginPassword password) {
        throw new AuthorizationFailedException(password.getLogin());

    }

    @Override
    public boolean shouldAuthorize(HttpServletRequest request, WebContext.LocalWebContext context) {
        if (request.getHeader(AUTH_TYPE_HEADER) == null || AUTH_TYPE_DEFAULT.equals(request.getHeader(AUTH_TYPE_HEADER))) {
            log.debug("Header AUTH-TYPE declares to use DefaultAuthorizationDelegator");


            if (context.getBean(ServerConfigRepo.class).filteredFirst(context
                    .getDataFilterProvider().eq("active", true).and(context.getDataFilterProvider().eq("type", ServerTypeProvider.LDAP))).orElse(null)!= null){
                throw new AuthorizationFailedException("LDAP CONFIG is active. Default authorization is not possible");
            }


            return true;
        }
        log.debug("Header AUTH-TYPE declares not to use DefaultAuthorizationDelegator");
        return false;
    }


    @Override
    public boolean blockUser(boolean block , Long userDataId, WebContext.LocalWebContext context) {
        var service = context.getBean( UserDataRepository.class);

        var item = service.findById(userDataId).orElseThrow(()-> ItemNotFoundException.fromId(userDataId));


        service.save(item);

        var activeTokens = context.getBean(ActiveTokensRepo.class).findByUserDataIdAndActiveIsTrueAndDueGreaterThan(userDataId, new Date());

        for (var token : activeTokens) {
            logout(token.getToken(), context, token.getScope());
        }

        return true;
    }

    @Override
    public boolean logout(String token, WebContext.LocalWebContext context, ScopeModel scope) {

        context.getBean(TokenProvider.class).removeAuthToken(token, ActiveToken.DEFAULT_TOKEN_TYPE, scope);

        return true;
    }

    @SneakyThrows
    @Override
    public boolean register(RegistrationForm form, WebContext.LocalWebContext context, ScopeModel scope, Long invitationId) {
        try {

            validateForm(form, context, scope, invitationId);


            InvitationModel invitationModel = null;
            if (invitationId != null && invitationId > 0) {
                invitationModel = context.getBean(InvitationService.class).findByIdOrThrow(invitationId);
                context.getBean(InvitationService.class).accept(invitationModel);
            }

            var service = context.getBean(LoginPasswordService.class);

            var newLoginPassword = getOrSavePassword(form, context);
            UserData userData = getOrSaveUser(form, context);

            if (Objects.nonNull(invitationModel)) {
                context.getBean(InvitationService.class).addUserToScope(invitationModel);
            } else {
                context.getBean(UserInScopeService.class).generateAndSave(userData, scope);
                service.saveUserRoles(newLoginPassword, scope);
            }

            return true;
        } catch (ValidationFailedException | StandardException ve){
            throw ve;
        }catch (Exception x) {
            log.error("error on save user data " + form.getEmail(), x);
            throw new AccessException("Cannot register", x);
        }

    }

    protected
    void validateForm(RegistrationForm form, WebContext.LocalWebContext context, ScopeModel scope, Long invitationId) {
        ((ValidateForm<RegistrationForm>) () -> context).validate(form, false, LoginPasswordService.class);
    }

    @SneakyThrows
    @Override
    public void validate(UserTokenDTO dto, WebContext.LocalWebContext context) {
        dto.validate(context);
        if (dto.getUserData().getBlocked() == Boolean.TRUE){
            throw new UserIsBlockedException("User is blocked");
        }
    }

    @Override
    public boolean changePassword(HttpServletRequest req, WebContext.LocalWebContext context, ChangePasswordForm loginPassword, ScopeModel scope) {
        if (!Objects.equals(loginPassword.getPassword(), loginPassword.getRepeatPassword())){
            throw new AuthorizationFailedException("global.exceptions.incorrect_password");
        }

        /****
         * prepare for authorization
         */
        LoginPassword lp = new LoginPassword();
        lp.setPassword(loginPassword.getCurrentPassword());
        lp.setLogin(context.getBean(SecurityUtils.class).get().getUserData().getLogin());
        lp.setTwoFactorCode(loginPassword.getTwoFactorCode());


        /**
         * authorize items
         */
        try{
            this.authorize(req, lp , context);
        }catch (Exception e){
            //user had wrong username / password
            throw new AuthorizationFailedException(lp.getLogin());
        }
        lp.setPassword(loginPassword.getPassword());
        context.getBean(ChangePasswordServiceImpl.class).changePassword(lp);
        return true;
    }

    @Override
    public String forgetPassword(HttpServletRequest req, WebContext.LocalWebContext context, String login) {

        return context.getBean(AuthenticationService.class).restore(login, context);

    }

    private LoginPassword getOrSavePassword(RegistrationForm form, WebContext.LocalWebContext context) {
        var service = context.getBean(LoginPasswordService.class);
        Optional<LoginPassword> loginPassword = service.findUserByLogin(form.getEmail());
        if (loginPassword.isPresent()) {
            return loginPassword.get();
        }

        var newLoginPassword = new LoginPassword();
        newLoginPassword.setLogin(form.getEmail());
        newLoginPassword.setPassword(form.getPassword());

        return service.save(newLoginPassword);
    }

    private UserData getOrSaveUser(RegistrationForm form, WebContext.LocalWebContext context) {
        Optional<UserData> user = context.getBean(ProfilePageService.class).findByEmail(form.getEmail());
        if (user.isPresent()) {
            return user.get();
        }
        UserData userData = context.getBean(LoginPasswordService.class).saveUserData(form);
        context.getBean(AuthScopeService.class).generateAndSaveUserScope(userData);
        return userData;
    }
}
