package systems.dennis.auth.controller;

import org.springframework.web.bind.annotation.*;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.exception.InvitationException;
import systems.dennis.auth.form.InvitationForm;
import systems.dennis.auth.model.InvitationModel;
import systems.dennis.auth.repository.InvitationRepo;
import systems.dennis.auth.service.InvitationService;
import systems.dennis.auth.service.ProfilePageService;
import systems.dennis.shared.annotations.WebFormsSupport;
import systems.dennis.shared.annotations.security.WithRole;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.items.AbstractSelfItemsListController;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.postgres.controller.AddItemController;
import systems.dennis.shared.postgres.controller.DeleteItemController;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.utils.ApplicationContext;

import java.util.ArrayList;
import java.util.List;

@RestController
@RequestMapping ("/api/v2/auth/invitation")
@WebFormsSupport(value = InvitationService.class)
@CrossOrigin
public class InvitationController
        extends ApplicationContext
        implements AddItemController<InvitationModel, InvitationForm>,
        AbstractSelfItemsListController<InvitationModel, InvitationForm, Long>,
        DeleteItemController<InvitationModel> {

    public InvitationController(WebContext context) {
        super(context);
    }

    @PostMapping("/accept/{id}")
    public Boolean accept(@PathVariable(value = "id") Long invitationId) {
        InvitationModel invitation = getService().findByIdOrThrow(invitationId);
        Boolean accepted = getService().accept(invitation);
        getService().addUserToScope(invitation);
        return accepted;
    }

    @GetMapping("/my_accepted")
    @WithRole
    public List<InvitationForm> getMyInvitation(@RequestParam(value = "from", required = false) Long from,
                                       @RequestParam(value = "limit", required = false) Integer limit,
                                       @RequestParam(value = "page", required = false) Integer page) {
        List<InvitationModel> models = getService().getMyAcceptInvitation(from, limit, page);
        List<InvitationForm> result = new ArrayList<>();

        for (InvitationModel model : models) {
            result.add(toForm(model));
        }

        return result;
    }

    @GetMapping("/info")
    public InvitationForm getInfo(@RequestParam(value = "id") Long id) {
        InvitationModel invitation = getService().findByIdOrThrow(id);
        if (invitation.getAccepted()) {
            throw new InvitationException("global.exceptions.invitation_already_accept");
        }
        return toForm(invitation);
    }

    @WithRole("ROLE_SYNC")
    @GetMapping ("/user/{userId}")
    public InvitationForm getByUser(@PathVariable(value = "userId") Long userId) {
        UserData user = getBean(ProfilePageService.class).findByIdOrThrow(userId);
        AbstractDataFilter<InvitationModel> specification = getService().getFilterImpl().eq("email", user.getEmail())
                .and(getService().getFilterImpl().eq("accepted", true));
        InvitationModel model = getBean(InvitationRepo.class).filteredFirst(specification).orElseThrow(() -> ItemNotFoundException.fromId(user.getEmail()));
        return toForm(model);
    }

    @Override
    public InvitationService getService() {
        return getBean(InvitationService.class);
    }
}
