package systems.dennis.auth.service;

import org.springframework.stereotype.Service;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.entity.UserInScopeModel;
import systems.dennis.auth.exception.DuplicateRelationException;
import systems.dennis.auth.form.UserInScopeForm;
import systems.dennis.auth.mail.MailSender;
import systems.dennis.auth.model.VerificationTokenModel;
import systems.dennis.auth.repository.UserInScopeRepo;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.annotations.DeleteStrategy;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.postgres.service.PaginationService;;
import systems.dennis.shared.repository.AbstractDataFilter;
import systems.dennis.shared.scopes.model.ScopeModel;

import java.util.*;

import static systems.dennis.shared.annotations.DeleteStrategy.DELETE_STRATEGY_PROPERTY;

@Service
@DataRetrieverDescription(model = UserInScopeModel.class, form = UserInScopeForm.class, repo = UserInScopeRepo.class)
@DeleteStrategy(value = DELETE_STRATEGY_PROPERTY)
public class UserInScopeService extends PaginationService<UserInScopeModel> {

    public UserInScopeService(WebContext holder) {
        super(holder);
    }

    @Override
    public UserInScopeModel afterAdd(UserInScopeModel object) {
        ScopeModel scope = object.getScope();

        sendVerificationEmail(scope, object.getUser());

        return super.afterAdd(object);
    }

    public void sendVerificationEmail(ScopeModel scope, UserData userData) {
        if (Objects.nonNull(scope.getScopeRule())) {
            if (scope.getScopeRule().getVerificationRequired() && (userData.getVerified() == null || !userData.getVerified())) {
                VerificationTokenModel verificationToken = getBean(VerificationTokenService.class).saveToken(userData, getVerificationTokenDuration(scope));
                generateConfirmationEmail(userData, verificationToken, scope);
            }
        }
    }

    public boolean isRelationExist(UserData user, ScopeModel scope) {
        AbstractDataFilter<UserInScopeModel> specification = getFilterImpl().eq("user", user)
                .and(getFilterImpl().eq("scope", scope));

        return count(specification) > 0;
    }

    public UserInScopeModel generateAndSave(UserData user, ScopeModel scope) {
        UserInScopeModel userInScope = new UserInScopeModel();
        userInScope.setUser(user);
        userInScope.setScope(scope);

        if (isRelationExist(user, scope)) {
            throw new DuplicateRelationException("global.exceptions.duplicate.user_in_scope");
        }

        return save(userInScope);
    }

    public List<UserInScopeModel> getByUser(Long userId) {
        UserData userData = getBean(ProfilePageService.class).findByIdOrThrow(userId);
        AbstractDataFilter<UserInScopeModel> specification = getFilterImpl().eq("user", userData);
        return getRepository().filteredData(specification).toList();
    }

    public UserInScopeModel getByUserAndScope(UserData user, ScopeModel scope) {
        AbstractDataFilter<UserInScopeModel> specification = getFilterImpl().eq("user", user)
                .and(getFilterImpl().eq("scope", scope));
        return getRepository().filteredFirst(specification).orElseThrow(() -> ItemNotFoundException.fromId(scope.getId()));
    }

    private void generateConfirmationEmail(UserData userData, VerificationTokenModel verificationToken, ScopeModel scope) {
        String url = scope.getVerificationUrl() + getVerificationScopePath() + "?token=" + verificationToken.getToken()
                + "&scope=" + scope.getName();
        String path = userData.getPreferredLanguage() + "/confirmation.html";

        Map<String, String> templateParameters = new HashMap<>();
        templateParameters.put("link", url);

        String content = getBean(MailSender.class).processHtmlTemplate(path, templateParameters);
        String title = getContext().getMessageTranslation("email.confirmation.title", userData.getPreferredLanguage());

        getBean(MailSender.class).sendMail(Collections.singletonList(userData.getEmail()), content, title);
    }

    private int getVerificationTokenDuration(ScopeModel scope) {
        if (Objects.nonNull(scope.getScopeRule())) {
            Integer tokenDuration = scope.getScopeRule().getVerificationTokenDuration();
            if (tokenDuration != null && tokenDuration > 0) {
                return tokenDuration;
            }
        }
        return getContext().getEnv("verification.token.duration.account-confirmation", 43200);
    }

    private String getVerificationScopePath() {
        return getContext().getEnv("app.verification.scope.path", "/verification/verify_scope");
    }
}
