package systems.dennis.auth.mail;

import com.sun.mail.smtp.SMTPSendFailedException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.Environment;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.HealthCheck;
import systems.dennis.shared.controller.ServerErrorResponse;
import systems.dennis.shared.utils.ApplicationContext;
import systems.dennis.shared.utils.SimpleEvaluator;
import systems.dennis.auth.exception.EmailSendingException;

import javax.mail.Authenticator;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;

import static systems.dennis.shared.controller.ServerErrorResponse.MAIL_AUTHENTICATION_ERROR;

@Component
@Slf4j
public class MailSender extends ApplicationContext {
    private final Environment environment;

    public MailSender(WebContext context, Environment environment) {
        super(context);
        this.environment = environment;
    }

    public void sendMail(List<String> recipients, String body, String title) {
        Properties properties = new Properties();
        properties.put("mail.smtp.host", getHost());
        properties.put("mail.smtp.port", getPort());
        properties.put("mail.smtp.auth", isAuth());
        properties.put("mail.smtp.starttls.enable", isStarttls());

        Session session = Session.getInstance(properties, new Authenticator() {
            @Override
            protected PasswordAuthentication getPasswordAuthentication() {
                return new PasswordAuthentication(getUsername(), getPassword());
            }
        });

        session.setDebug(true);

        try {
            MimeMessage message = new MimeMessage(session);
            message.setFrom(new InternetAddress(getUsername()));

            List<InternetAddress> recipientAddresses = new ArrayList<>();
            for (String recipient : recipients) {
                recipientAddresses.add(new InternetAddress(recipient));
            }
            message.setRecipients(Message.RecipientType.TO, recipientAddresses
                    .toArray(new InternetAddress[recipientAddresses.size()]));
            message.setSubject(title);
            message.setContent(body, "text/html; charset=UTF-8");

            Transport.send(message);
        } catch (SMTPSendFailedException e) {
            if (e.getReturnCode() == 502) {
                HealthCheck.errors.add(new ServerErrorResponse(e.getReturnCode(), MAIL_AUTHENTICATION_ERROR, e.getMessage()));
            }
            throw new EmailSendingException(e, "Error sending email");
        } catch (MessagingException e) {
            log.error("Error sending email");
            throw new EmailSendingException(e, "Error sending email");
        }
    }

    public String processHtmlTemplate(String path, Map<String, String> params) {
        try {
            Resource resource = new ClassPathResource("mail-templates/" + path);
            InputStream inputStream = resource.getInputStream();
            String html = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))
                    .lines()
                    .collect(Collectors.joining("\n"));
            SimpleEvaluator evaluator = new SimpleEvaluator(environment);
            return evaluator.evaluate(html, 0, params);
        } catch (IOException e) {
            throw new IllegalStateException("Failed to process HTML template: " + path, e);
        }
    }

    private String getHost() {
        return getContext().getEnv("app.auth.mail.host");
    }

    private int getPort() {
        return getContext().getEnv("app.auth.mail.port");
    }

    private String getUsername() {
        return getContext().getEnv("app.auth.mail.username");
    }

    private String getPassword() {
        return getContext().getEnv("app.auth.mail.password");
    }

    private boolean isAuth() {
        return getContext().getEnv("app.auth.mail.properties.mail.smtp.auth");
    }

    private boolean isStarttls() {
        return getContext().getEnv("app.auth.mail.properties.mail.smtp.starttls.enable");
    }
}
