package systems.dennis.auth.client.utils;

import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import org.springframework.stereotype.Service;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.exception.NeedAuthorizationException;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.UserServiceImpl;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.entity.TokenData;
import systems.dennis.shared.exceptions.AuthorizationNotFoundException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.utils.Supplier;

import java.util.stream.Collectors;

import static systems.dennis.auth.config.AuthorizationDelegator.AUTH_SCOPE_HEADER;


@Service
@Slf4j
public class AuthenticationService {
    public static final String AUTHORIZATION_HEADER = "Authorization";
    private final WebContext context;
    private boolean authorized = false;

    public AuthenticationService(WebContext context) {
        this.context = context;
    }

    public UserTokenDTO authorize(LoginPassword loginPassword, ScopeModel scope) {
        if (authorized) {
            logout();
        }

        var res = context.getBean(UserServiceImpl.class).authorize(loginPassword, scope);

        authorized = true;
        return res;
    }

    public UserTokenDTO authorizeVirtual(LoginPassword loginPassword, ScopeModel scope) {
        if (authorized) {
            logout();
        }

        var res = context.getBean(UserServiceImpl.class).authorizeVirtual(loginPassword, scope);

        authorized = true;
        return res;
    }

    public boolean logout() {
        try {
            if (!authorized) {
                return false;
            }
            authorized = false;
            return true;
        } catch (Exception e) {
            log.error("Something wrong goes with logout: ", e);
            return false;
        }
    }


    boolean isAuthorized() {
        return authorized;
    }

    public UserTokenDTO get() throws NeedAuthorizationException {
        throw new NeedAuthorizationException();
    }



    public TokenData getToken(HttpServletRequest request) {

        if (request == null) {
            return null;
        }
        String token = request.getHeader(AUTHORIZATION_HEADER);
        String tokenScope = request.getHeader(AUTH_SCOPE_HEADER);

        if (token == null) {
            token = get((obj) -> new AuthorizationNotFoundException("Expected token, instead no token")).getToken();

        }

        token = token.replace("Bearer ", "").trim();

        return new TokenData( tokenScope, token);
    }

    @SneakyThrows
    private UserTokenDTO get(Supplier<? extends Exception> o) {
        try {
            return get();
        } catch (NeedAuthorizationException e) {
            throw o.onNull(e);
        }
    }


    @SneakyThrows
    public String restore(String login, WebContext.LocalWebContext context) {


        var userRepo = context.getBean(UserDataRepository.class);
        var passwordService = context.getBean(PasswordService.class);

        var dta = userRepo.findByLogin(login).orElseThrow(() -> ItemNotFoundException.fromId(login));

        return passwordService.reset(dta);

    }

    @SneakyThrows
    public boolean hasRole(String role) {
        return get().getRoleList().contains(role);
    }
}
