package systems.dennis.auth.service;

import org.apache.commons.codec.binary.Base64;

import org.springframework.stereotype.Service;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.form.VerificationTokenForm;
import systems.dennis.shared.annotations.DataRetrieverDescription;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.postgres.service.PaginationService;
import systems.dennis.auth.exception.VerificationTokenException;
import systems.dennis.auth.model.VerificationTokenModel;
import systems.dennis.auth.repository.VerificationTokenRepo;
import systems.dennis.shared.repository.AbstractDataFilter;

import java.nio.charset.StandardCharsets;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.UUID;

@Service
@DataRetrieverDescription(model = VerificationTokenModel.class, form = VerificationTokenForm.class, repo = VerificationTokenRepo.class)
public class VerificationTokenService extends PaginationService<VerificationTokenModel> {
    private static final int KEY_LENGTH = 15;

    public VerificationTokenService(WebContext holder) {
        super(holder);
    }

    public VerificationTokenModel saveToken(UserData userData, int duration) {
        VerificationTokenModel model = new VerificationTokenModel();
        String token = generateToken();

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.MINUTE, duration);

        model.setToken(token);
        model.setExpirationDate(calendar.getTime());
        model.setUserData(userData);

        return save(model);
    }

    public void validateVerificationToken(String token) {
        VerificationTokenModel verificationToken = getByToken(token);
        Date currentDate = new Date();

        if (currentDate.after(verificationToken.getExpirationDate())) {
            throw new VerificationTokenException("global.exceptions.token.expired");
        }
    }

    public void deleteToken(String token) {
        VerificationTokenModel verificationToken = getByToken(token);
        delete(verificationToken.getId());
    }

    public VerificationTokenModel getByToken(String token) {
        return getRepository().filteredOne(getFilterImpl().eq("token", token))
                .orElseThrow(()-> ItemNotFoundException.fromId(token));
    }

    public void deleteActiveTokens(UserData userData) {
        AbstractDataFilter<VerificationTokenModel> specification = getFilterImpl().eq("userData", userData)
                .and(getFilterImpl().greatEQ("expirationDate", new Date()));

        List<VerificationTokenModel> tokens = getRepository().filteredData(specification).toList();

        for (VerificationTokenModel token : tokens) {
            getRepository().delete(token);
        }
    }


    private String generateToken() {
        return new String(Base64.encodeBase64URLSafe(UUID.randomUUID().toString().getBytes()), StandardCharsets.US_ASCII);
    }
}
