package systems.dennis.auth.service;


import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.entity.ActiveToken;
import systems.dennis.auth.entity.LoginHistory;
import systems.dennis.auth.repository.LoginHistoryRepository;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.role_validator.TokenProvider;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.exceptions.AccessDeniedException;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.scopes.model.ScopeModel;

import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Default user Authorization and management class
 */

@Service
@Slf4j
public class UserServiceImpl {

    private final UserDataRepository userRepository;

    private final TokenProvider tokenProvider;
    private final LoginHistoryRepository historyRepository;

    private final PasswordService passwordService;
    private final LoginPasswordService service;


    public UserServiceImpl(UserDataRepository userRepository,
                           TokenProvider tokenProvider,
                           LoginHistoryRepository historyRepository, PasswordService passwordService, LoginPasswordService service) {
        this.userRepository = userRepository;
        this.tokenProvider = tokenProvider;

        this.historyRepository = historyRepository;
        this.passwordService = passwordService;
        this.service = service;
    }


    public UserTokenDTO authorize(LoginPassword password, ScopeModel scope) {
        LoginPassword res = null;
        password.setPassword(passwordService.toPassword(password.getPassword()));
        try {
            res = service.findOrThrow(password, true);
            if (res == null) throw new AccessDeniedException(password.getLogin());
        } catch (AccessDeniedException e) {
            //In order not to move to login page, and throw correct exception, we will throw a separate exception which will
            //throw 401 status on error
            throw new AuthorizationFailedException(password.getLogin());
        }

        UserData userData = createNew(res);
        UserTokenDTO dto = new UserTokenDTO();


        dto.setUserData(userData);
        log.debug("Start authorizing , creating token: {}",password.getLogin());
        var roles = passwordService.getRoles(res, scope);
        var token = tokenProvider.createToken(dto, ActiveToken.DEFAULT_TOKEN_TYPE, roles, scope);
        dto.setToken(token.getToken());
        dto.setRoles(roles.stream().map(UserRole::getRole).collect(Collectors.toList()));
        dto.setDue(token.getDue());

        LoginHistory history = new LoginHistory();
        log.debug("TRacing Login history started");
        history.setUserDataId(userData.getId());
        history.setToken(token.getToken());
        history.setLogin(userData.getLogin());
        historyRepository.save(history);
        log.debug("TRacing Login history finished");

        return dto;
    }

    public UserTokenDTO authorizeVirtual(LoginPassword password, ScopeModel scope) {
        LoginPassword res = null;
        password.setPassword(passwordService.toPassword(password.getPassword()));
        try {
            res = service.findOrThrow(password, false);
            if (res == null) throw new AccessDeniedException(password.getLogin());
        } catch (AccessDeniedException e) {
            //In order not to move to login page, and throw correct exception, we will throw a separate exception which will
            //throw 401 status on error
            throw new AuthorizationFailedException(password.getLogin());
        }

        UserData userData = createNew(res);
        UserTokenDTO dto = new UserTokenDTO();


        dto.setUserData(userData);
        log.debug("Start authorizing , creating token: {}",password.getLogin());
        var roles = passwordService.getRoles(res, scope);
        var token = tokenProvider.createToken(dto, ActiveToken.DEFAULT_VIRTUAL_TYPE, roles, scope);
        dto.setToken(token.getToken());
        dto.setRoles(roles.stream().map(UserRole::getRole).collect(Collectors.toList()));
        dto.setDue(token.getDue());

        LoginHistory history = new LoginHistory();
        log.debug("TRacing Login history started");
        history.setUserDataId(userData.getId());
        history.setToken(token.getToken());
        history.setLogin(userData.getLogin());
        historyRepository.save(history);
        log.debug("Tracing Login history finished");

        return dto;
    }

    @SneakyThrows
    private UserData createNew(LoginPassword principal) {

        final UserData temp = new UserData();

        temp.setLogin(principal.getLogin());

        Optional<UserData> user = userRepository.findByLogin(temp.getLogin());
        return user.orElseGet(() -> createUser(temp));
    }

    private UserData createUser(UserData temp) {

        UserData newUserData = new UserData();
        newUserData.setLogin(temp.getLogin());
        newUserData.setEmail(temp.getLogin());
        userRepository.save(newUserData);
        return newUserData;

    }

}
