package systems.dennis.auth.delegations.phone;

import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.TwillioClient;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.config.AuthorizeResponse;
import systems.dennis.auth.delegations.simple.DefaultAuthorizationDelegator;
import systems.dennis.auth.delegations.simple.SimplePhoneAuthorization;
import systems.dennis.auth.form.ChangePasswordForm;
import systems.dennis.auth.form.RegistrationForm;
import systems.dennis.auth.model.PhoneAuthorizationCodes;
import systems.dennis.auth.model.TempAccount;
import systems.dennis.auth.repository.PhoneAuthorizationCodeRepo;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.service.TempAccountService;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.AccessDeniedException;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.exceptions.StatusException;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.scopes.service.ScopeService;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.servers.repository.ServerConfigRepo;

import java.util.Calendar;
import java.util.Date;

@Slf4j
public class PhoneAuthorizationDelegator extends DefaultAuthorizationDelegator {
    public static final String AUTH_TYPE_PHONE = "phone";
    @Override
    public AuthorizeResponse authorize(HttpServletRequest request, LoginPassword password, WebContext.LocalWebContext context) {
         context.getBean(PhoneAuthorizationCodeRepo.class).filteredFirst(context.getDataFilterProvider().eq("login", password.getLogin())
                .and(context.getDataFilterProvider().eq("code", password.getPassword()))
                .and(context.getDataFilterProvider().greater("expiryDate", new Date())

        )).orElseThrow(()-> new AuthorizationFailedException(password.getPassword()));
        var userData = context.getBean(UserDataRepository.class).filteredFirst(context.getDataFilterProvider().eq("login", password.getLogin())).orElseThrow(()->ItemNotFoundException.fromId(password.getLogin()));
        var scope = context.getBean(ScopeService.class).findByName(request.getHeader(AUTH_SCOPE_HEADER), userData.getId(), true);
        if (userData.getBlocked()) {
            throw new AuthorizationFailedException("global.app.user_blocked");
        }
        var auth =  new SimplePhoneAuthorization().authorize(password, context, scope);

        AuthorizeResponse resp = new AuthorizeResponse();
        resp.setSuccess(true);
        resp.setDate(new Date());
        resp.setDto(auth);
        resp.setWithOldToken(false);
        return resp;

    }

    @Override
    public boolean shouldAuthorize(HttpServletRequest request, WebContext.LocalWebContext context) {
        if (request.getHeader(AUTH_TYPE_HEADER) != null && AUTH_TYPE_PHONE.equals(request.getHeader(AUTH_TYPE_HEADER))) {
            log.debug("Header AUTH-TYPE declares to use DefaultAuthorizationDelegator");


            if (context.getBean(ServerConfigRepo.class).filteredFirst(context
                    .getDataFilterProvider().eq("active", true).and(context.getDataFilterProvider().eq("type", ServerTypeProvider.LDAP))).orElse(null)!= null){
                throw new AuthorizationFailedException("LDAP CONFIG is active. Default authorization is not possible");
            }


            return true;
        }
        log.debug("Header AUTH-TYPE declares not to use DefaultAuthorizationDelegator");
        return false;
    }

    @Override
    protected void validateForm(RegistrationForm form, WebContext.LocalWebContext context, ScopeModel scope, Long invitationId) {
        //super.validateForm(form, context, scope, invitationId);
    }

    @Override
    public boolean register(RegistrationForm form, WebContext.LocalWebContext context, ScopeModel scope, Long invitationId) {

        var service =context.getBean(TempAccountService.class);

        var tempAccount = service.getRepository().filteredFirst(context.getDataFilterProvider().eq("login", form.getEmail())).orElseThrow(()-> ItemNotFoundException.fromId(form.getEmail()));

        if (tempAccount.getCode().equalsIgnoreCase(form.getPassword())){
            var newPwd = context.getBean(PasswordService.class).generateRandomPassword(15);

            form.setPassword(newPwd);
            form.setRepeatPassword(newPwd);
            super.register(form, context, scope, invitationId);
            tempAccount.setEnabled(true);
            service.save(tempAccount);
            return true;
        } else {
            throw new StatusException("user.not_verified.error", HttpStatus.BAD_REQUEST);
        }
    }

    @Override
    public boolean changePassword(HttpServletRequest req, WebContext.LocalWebContext context, ChangePasswordForm loginPassword, ScopeModel scopeModel) {
        throw new AccessDeniedException("global.phone_authorization_is_not_able_to_change_password");
    }

    @Override
    public String forgetPassword(HttpServletRequest req, WebContext.LocalWebContext context, String login) {
        throw new AccessDeniedException("global.phone_authorization_is_not_able_to_restore");
    }

    @Override
    public boolean requestAuthorization(HttpServletRequest req, WebContext.LocalWebContext context, String login) {

        var scopeName = req.getHeader(AUTH_SCOPE_HEADER);
        var scopeService = context.getBean(ScopeService.class);

        var scopeModel = scopeService.getRepository().filteredFirst(scopeService.getFilterImpl().eq("name", scopeName)).orElseThrow(() -> ItemNotFoundException.fromId(scopeName));

        if (scopeModel.getScopeRule() == null || !scopeModel.getScopeRule().getAllowPhoneOperations()) {
            throw new AuthorizationFailedException("global.scope_does_not_allow_phone_operations");
        }

        var userData = context.getBean(UserDataRepository.class).findByLogin(login).orElseThrow(() -> ItemNotFoundException.fromId(login));

        if (userData.getBlocked()) {
            throw new AccessDeniedException("global.user_is_blocked");
        }


        var code = createCode(login, context, "authorization");

        sendCode("request_authorization", code, userData, context);

        return true;
    }
    @Override
    public boolean requestRegistration(HttpServletRequest req, WebContext.LocalWebContext context, String login) {

        login = login.trim();
        var scopeName = req.getHeader(AUTH_SCOPE_HEADER);
        var scopeService = context.getBean(ScopeService.class);

        var scopeModel = scopeService.getRepository().filteredFirst(scopeService.getFilterImpl().eq("name", scopeName)).orElseThrow(() -> ItemNotFoundException.fromId(scopeName));

        if (scopeModel.getScopeRule() == null || !scopeModel.getScopeRule().getAllowPhoneOperations() || !scopeModel.getScopeRule().getRegistrationAllowed()) {
            throw new AuthorizationFailedException("global.scope_does_not_allow_phone_operations_to_register");
        }

        var userData = context.getBean(UserDataRepository.class).findByLogin(login).orElse(null);

        if (userData != null) {
            throw new AccessDeniedException("global.user_is_already_registered");
        }

        var serviceTempAccount = context.getBean(TempAccountService.class);

        var existing = serviceTempAccount.getRepository().exists(context.getDataFilterProvider().eq("login", login));
        if (existing){
            throw new StatusException("global.app_request_was_already_done", HttpStatus.BAD_REQUEST);
        }
        var code = createCode(login, context, "register");

        TempAccount account = new TempAccount();
        account.setLogin(login);
        account.setCode(code);
        account.setEnabled(false);
        serviceTempAccount.save(account);



        UserData data = new UserData();
        data.setLogin(login);

        sendCode("request_registration", code, data, context);



        return true;
    }

    public void sendCode(String requestAuthorization, String code, UserData userData, WebContext.LocalWebContext context) {
        TwillioClient client = new TwillioClient(context.getWebContext());
        client.send(getMessage(code, userData ,  context), code);

    }

    public String getMessage(String code, UserData userData, WebContext.LocalWebContext context){
        return "code is " + code;
    }

    public String createCode(String login, WebContext.LocalWebContext context, String register) {

        var repo = context.getBean(PhoneAuthorizationCodeRepo.class);
        var exisitng = repo
                .filteredFirst(context.getDataFilterProvider().eq("login", login)
                        .and(context.getDataFilterProvider().greater("expiryDate", new Date()))

                )

                .orElseGet(PhoneAuthorizationCodes::new);

        if (exisitng.getId() != null) {
            return exisitng.getCode();
        }

        exisitng.setCode(PasswordService.generateRandomKey(6, "0123456789"));
        exisitng.setExpiryDate(getExpireDate(context));
        exisitng.setLogin(login);
        repo.save(exisitng);
        return exisitng.getCode();


    }

    public Date getExpireDate(WebContext.LocalWebContext context) {
        var calendar = Calendar.getInstance();
        calendar.add(Calendar.MINUTE, context.getEnv("global.app.phone_sms_duration_in_minutes", 15));
        return calendar.getTime();
    }
}
