package systems.dennis.auth.delegations.ldap;

import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.config.AuthorizationDelegator;
import systems.dennis.auth.config.AuthorizeResponse;
import systems.dennis.auth.delegations.simple.AuthorizationAttemptProcessor;
import systems.dennis.auth.entity.ActiveToken;
import systems.dennis.auth.entity.LoginHistory;
import systems.dennis.auth.form.ChangePasswordForm;
import systems.dennis.auth.form.RegistrationForm;
import systems.dennis.auth.repository.ActiveTokensRepo;
import systems.dennis.auth.repository.LoginHistoryRepository;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.role_validator.TokenProvider;
import systems.dennis.auth.role_validator.entity.UserRole;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.service.AuthScopeService;
import systems.dennis.auth.service.LoginPasswordService;
import systems.dennis.auth.service.ProfilePageService;
import systems.dennis.auth.service.UserInScopeService;
import systems.dennis.auth.util.PasswordService;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.AccessDeniedException;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.servers.model.ServerConfig;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.servers.repository.ServerConfigRepo;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import java.util.Date;
import java.util.Hashtable;
import java.util.stream.Collectors;

@Slf4j
public class LdapAuthorization implements AuthorizationAttemptProcessor, AuthorizationDelegator {
    @Override
    public <T extends UserTokenDTO> T authorize(LoginPassword loginPassword, WebContext.LocalWebContext context, ScopeModel scope) {


        var config = (ServerConfig) context.getBean(ServerConfigRepo.class).filteredFirst(context
                .getDataFilterProvider().eq("active", true).and(context.getDataFilterProvider().eq("type", ServerTypeProvider.LDAP))).orElse(null);;

        if (config == null) {
            log.info(" NO LDAP CONFIG FOUND. return null");
            return null;
        }

        if (loginPassword.getDomain() == null) {
            throw new AuthorizationFailedException("Domain is required for authorization");
        }


        // set up the LDAP parameters
        Hashtable<Object, Object> env = new Hashtable<Object, Object>();
        env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
        env.put(Context.PROVIDER_URL, "ldap://" + config.getHost() + ":" + config.getPort());
        env.put(Context.SECURITY_AUTHENTICATION, "simple");
        env.put(Context.REFERRAL, "follow");
        env.put(Context.SECURITY_PRINCIPAL, loginPassword.getDomain() + "\\" + loginPassword.getLogin());
        env.put(Context.SECURITY_CREDENTIALS, loginPassword.getPassword());




        // attempt to authenticate
        InitialLdapContext ctx = null;
        try {
            ctx = new InitialLdapContext(env, null);
            var res = getUserInfo(loginPassword.getLogin(), ctx, getSearchControls(), config, context);
            if (!context.getBean(UserInScopeService.class).isRelationExist(res, scope)) {
                throw new AuthorizationFailedException(res.getLogin());
            }
            ctx.close();
            var lps = context.getBean(LoginPasswordService.class);

            var existingLoginPassword = lps.findUserByLogin(loginPassword.getLogin());
            if (existingLoginPassword.isPresent()){
                loginPassword = existingLoginPassword.get();
            } else{
                loginPassword = lps.save(loginPassword);
            }
            UserTokenDTO dto = new UserTokenDTO();
            dto.setUserData(res);
            var roles = context.getBean(PasswordService.class).getRoles(loginPassword, scope);
            var token = context.getBean(TokenProvider.class).createToken(dto, ActiveToken.DEFAULT_LDAP_TOKEN_TYPE, roles, scope);
            dto.setToken(token.getToken());
            dto.setDue(token.getDue());
            dto.setRoles(roles.stream().map(UserRole::getRole).collect(Collectors.toList()));


            return (T) dto;
        } catch (NamingException e) {
            throw new AuthorizationFailedException(loginPassword.getLogin());
        }
    }

    private static SearchControls getSearchControls() {
        SearchControls cons = new SearchControls();
        cons.setSearchScope(SearchControls.SUBTREE_SCOPE);
        String[] attrIDs = {"distinguishedName", "sn", "givenname", "mail", "telephonenumber", "thumbnailPhoto"};
        cons.setReturningAttributes(attrIDs);
        cons.setDerefLinkFlag(true);
        return cons;
    }


    private UserData getUserInfo(String userName, InitialLdapContext ctx, SearchControls
            searchControls, ServerConfig config, WebContext.LocalWebContext context) {
        System.out.println("*** " + userName + " ***");
        UserData user = context.getBean(UserDataRepository.class).findByLogin(userName).orElse( new UserData());
        try {
            NamingEnumeration<SearchResult> answer = ctx.search(config.getServerParam(), "sAMAccountName=" + userName, searchControls);
            if (answer.hasMore()) {
                Attributes attrs = answer.next().getAttributes();

                user.setLogin(userName);
                user.setEmail(getValue(attrs.get("mail")));
                user.setName(getValue(attrs.get("givenname")) + " " + getValue(attrs.get("sn")));
                user.setPhone(getValue(attrs.get("telephonenumber")));
                user.setEmail(getValue(attrs.get("mail")));
                user = context.getBean(UserDataRepository.class).save(user);
            } else {
                throw new AuthorizationFailedException("User had successfully logged in, but there is no info in LDAP about user" + userName );
            }
            answer.close();
        } catch (AuthorizationFailedException e) {
            throw e;
        }catch (Exception ex) {
            ex.printStackTrace();
        }



        return user;
    }

    private String getValue(Attribute attribute) {
        try {
            return String.valueOf(attribute.get());
        } catch (Exception e) {
            return "";
        }
    }


    @Override
    public <T extends UserData> T createUser(LoginPassword loginPassword, WebContext.LocalWebContext context) {
        return null;
    }

    @Override
    public void saveLoginAttempt(UserTokenDTO user, WebContext.LocalWebContext context) {
        LoginHistory history = new LoginHistory();
        log.debug("TRacing Login history started");
        history.setUserDataId(context.getCurrentUser());
        history.setToken(user.getToken());
        history.setAuthorizationType(ActiveToken.DEFAULT_LDAP_TOKEN_TYPE);
        history.setLogin(user.getUserData().getLogin());
        context.getBean(LoginHistoryRepository.class).save(history);
    }

    @Override
    public AuthorizeResponse authorize(HttpServletRequest request, LoginPassword password, WebContext.LocalWebContext context) {
        UserData user = context.getBean(ProfilePageService.class).findByLogin(password.getLogin()).orElseThrow(() -> ItemNotFoundException.fromId(password.getLogin()));
        ScopeModel scope = context.getBean(AuthScopeService.class).getScopeFromRequest(request, user.getId(), false);
        context.getBean(ProfilePageService.class).checkVerifiedUser(user, scope);

        var res = this.authorize(password, context, scope);
        return AuthorizeResponse.of( res, false);
    }

    @Override
    public boolean shouldAuthorize(HttpServletRequest request, WebContext.LocalWebContext context) {
        if ( ActiveToken.DEFAULT_LDAP_TOKEN_TYPE.equals(request.getHeader(AUTH_TYPE_HEADER))) {
            log.debug("Header AUTH-TYPE declares to use VirtualUserAuth");
            return true;
        }
        log.debug("Header AUTH-TYPE declares not to use VirtualUserAuth");
        return false;
    }


    @Override
    public boolean blockUser(boolean block, Long userDataId, WebContext.LocalWebContext context) {

        var service = context.getBean( UserDataRepository.class);

        var item = service.findById(userDataId).orElseThrow(()-> ItemNotFoundException.fromId(userDataId));


        service.save(item);

        var activeTokens = context.getBean(ActiveTokensRepo.class).findByUserDataIdAndActiveIsTrueAndDueGreaterThan(userDataId, new Date());

        for (var token : activeTokens) {
            logout(token.getToken(), context, token.getScope());
        }

        return true;

    }

    @Override
    public boolean logout(String token,  WebContext.LocalWebContext context, ScopeModel scope) {
        context.getBean(TokenProvider.class).removeAuthToken(token, ActiveToken.DEFAULT_VIRTUAL_TYPE, scope);
        return true;
    }

    @Override
    public boolean register(RegistrationForm form, WebContext.LocalWebContext context, ScopeModel scope, Long invitationId) {
        throw new AccessDeniedException("User can not be registered in LDAP");
    }

    @SneakyThrows
    @Override
    public void validate(UserTokenDTO dto, WebContext.LocalWebContext context) {
        dto.validate(context);
    }

    @Override
    public boolean changePassword(HttpServletRequest req, WebContext.LocalWebContext context, ChangePasswordForm loginPassword, ScopeModel scope) {
        throw new AccessDeniedException("User can not be changed in LDAP");

    }

    @Override
    public String forgetPassword(HttpServletRequest req, WebContext.LocalWebContext context, String login) {
        throw new AccessDeniedException("User can not restore password in LDAP");
    }


}
