package systems.dennis.auth.controller;

import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.entity.UserDataForm;
import systems.dennis.auth.entity.UserInScopeModel;
import systems.dennis.auth.form.RoleForm;
import systems.dennis.auth.provider.UserStatusProvider;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.service.AuthScopeService;
import systems.dennis.auth.service.LoginPasswordService;
import systems.dennis.auth.service.ProfilePageService;
import systems.dennis.auth.service.UserInScopeService;
import systems.dennis.auth.validators.UserDataChecker;
import systems.dennis.shared.annotations.WebFormsSupport;
import systems.dennis.shared.annotations.security.Id;
import systems.dennis.shared.annotations.security.Secured;
import systems.dennis.shared.annotations.security.SelfOnlyRole;
import systems.dennis.shared.annotations.security.WithRole;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.items.magic.MagicRequest;
import systems.dennis.shared.exceptions.*;
import systems.dennis.shared.postgres.controller.EditItemController;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.scopes.model.ScopeRuleModel;
import systems.dennis.shared.scopes.service.ScopeService;
import systems.dennis.shared.utils.ApplicationContext;
import systems.dennis.shared.utils.bean_copier.BeanCopier;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

import static systems.dennis.shared.form.AbstractForm.ID_FIELD;

@RestController
@CrossOrigin ("*")
@Secured
@RequestMapping ("/api/v2/auth/profile")
@WebFormsSupport(ProfilePageService.class)
public class UserDataProfileController extends ApplicationContext
        implements EditItemController<UserData, UserDataForm> {

    public UserDataProfileController(WebContext context) {
        super(context);
    }

    @PostMapping ("/update/photo/{id}")
    @WithRole ()
    public UserData create(@RequestParam("url") String url, @PathVariable("id") Long profileId ){
        return getBean(ProfilePageService.class).savePhoto( profileId, url);
    }

    @WithRole("ROLE_SYNC")
    @GetMapping ("/all/scope/{name}")
    public List<UserDataForm> findByScope(@PathVariable("name") String scopeName,
                                          @RequestParam(value = "time", required = false) @DateTimeFormat(pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS") String time) {
        var copier = getContext().getBean(BeanCopier.class);
        ScopeModel scope = getBean(ScopeService.class).findByName(scopeName, getCurrentUser(), true);

        MagicRequest request = new MagicRequest();
        request.getCases().add(getService().getFilterImpl().eq("scope", scope));

        if (Objects.nonNull(time)) {
            try {
                SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
                Date parseTime = dateFormat.parse(time);
                request.getCases().add(getService().getFilterImpl().greatEQ("lastUpdate", parseTime).setJoinOn("user").setComplex(true));
            } catch (Exception e) {
                throw new TransformException("global.exception.parsing_date");
            }
        }

        List<UserData> res = getBean(UserInScopeService.class).search(request).getContent()
                .stream()
                .map(UserInScopeModel::getUser)
                .collect(Collectors.toList());

        List<UserDataForm> result = new ArrayList<>();

        for (UserData data : res){
            var element = copier.copy(data, UserDataForm.class);
            element.setHiddenBySystem(getBean(ProfilePageService.class).isObjectDeleted(data));
            result.add(element);
        }
        return result;
    }

    @GetMapping ("/status/{mail}")
    public Long getUserStatus(@PathVariable("mail") String mail) {
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(getContext().getRequest(), null, true);
        ScopeRuleModel scopeRule = scope.getScopeRule();

        Optional<UserData> userData = getBean(ProfilePageService.class).findByEmail(mail);
        if (userData.isEmpty()) {
            return UserStatusProvider.NOT_REGISTERED;
        }
        if (Objects.nonNull(userData.get().getVerified()) && userData.get().getVerified()) {
            return UserStatusProvider.REGISTERED_AND_VERIFIED;
        }
        if (Objects.nonNull(scopeRule) && Objects.nonNull(scopeRule.getVerificationRequired()) && !scopeRule.getVerificationRequired()) {
            return UserStatusProvider.REGISTERED_NO_VERIFICATION_REQUIRED;
        }
        
        return UserStatusProvider.REGISTERED_NOT_VERIFIED;
    }

    @GetMapping ("/email/{mail}")
    public UserDataForm getUserByEmail(@PathVariable("mail") String mail) {
        UserData user = getBean(ProfilePageService.class).findByEmail(mail).orElseThrow(() -> ItemNotFoundException.fromId(mail));
        return  toForm(user);
    }

    @WithRole
    @PostMapping ("/update_lang/{lang}")
    public ResponseEntity changeLang(@PathVariable("lang") String lang){

        var user = getBean(UserDataRepository.class).findById(getCurrentUser()).orElseThrow(()-> ItemNotFoundException.fromId("Current user is not found on db"));
        user.setPreferredLanguage(lang);
        user.setLastUpdate(new Date());
        getBean(UserDataRepository.class).save(user);

        return ResponseEntity.ok().build();
    }

    @DeleteMapping ("/delete/{id}")
    @WithRole (RoleForm.ROLE_ADMIN)
    @ResponseBody
    public void delete( @PathVariable("id") Long profileId ){
        try {

            var passwordBean = getBean(LoginPasswordService.class);
            var obj = passwordBean.findByIdOrThrow(profileId);
            passwordBean.delete(profileId);


            var profileBean =  getBean(ProfilePageService.class);
            var profile = profileBean.findByLogin(obj.getLogin()).orElseThrow(()-> ItemNotFoundException.fromId(obj.getLogin()));

            getBean(ProfilePageService.class).delete(profile.getId());
        } catch (Exception e){
            throw new DeleteNotPossibleException();
        }
    }

    @Override
    @WithRole
    @SelfOnlyRole(userIdField = ID_FIELD)
    public ResponseEntity<UserDataForm> edit(@Id(checker = UserDataChecker.class) UserDataForm form) throws ItemForAddContainsIdException, ItemDoesNotContainsIdValueException, UnmodifiedItemSaveAttemptException, ItemNotFoundException {
        return EditItemController.super.edit(form);
    }
}
