package systems.dennis.auth.controller;

import jakarta.servlet.http.HttpServletRequest;
import org.springframework.web.bind.annotation.*;
import systems.dennis.auth.client.LoginPassword;
import systems.dennis.auth.client.entity.UserData;
import systems.dennis.auth.client.utils.SecurityUtils;
import systems.dennis.auth.config.AuthorizationDelegator;
import systems.dennis.auth.config.AuthorizeResponse;
import systems.dennis.auth.delegations.ldap.LdapAuthorization;
import systems.dennis.auth.delegations.simple.DefaultAuthorizationDelegator;
import systems.dennis.auth.delegations.virtual.VirtualUserAuthorizationDelegator;
import systems.dennis.auth.entity.ActiveToken;
import systems.dennis.auth.exception.*;
import systems.dennis.auth.form.ChangePasswordForm;
import systems.dennis.auth.form.RegistrationForm;
import systems.dennis.auth.mail.MailSender;
import systems.dennis.auth.model.VerificationTokenModel;
import systems.dennis.auth.repository.UserDataRepository;
import systems.dennis.auth.responses.Auth2FactorEnabled;
import systems.dennis.auth.service.*;
import systems.dennis.shared.annotations.security.WithRole;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.AuthorizationNotFoundException;
import systems.dennis.shared.exceptions.ItemNotFoundException;
import systems.dennis.shared.scopes.model.ScopeModel;
import systems.dennis.shared.utils.ApplicationContext;

import java.util.*;

import static systems.dennis.auth.config.AuthorizationDelegator.AUTH_TYPE_HEADER;


@RestController
@CrossOrigin(origins = "*")
@RequestMapping ("/api/v3/auth")
public class AuthorizeControllerVersion2 extends ApplicationContext {

    private static final Map<String, AuthorizationDelegator> delegatorMap = new HashMap<>();




    static {
        registerAuthorizationDelegator(DefaultAuthorizationDelegator.AUTH_TYPE_DEFAULT, new DefaultAuthorizationDelegator());
        registerAuthorizationDelegator(VirtualUserAuthorizationDelegator.AUTH_TYPE_VIRTUAL, new VirtualUserAuthorizationDelegator());
        registerAuthorizationDelegator(ActiveToken.DEFAULT_LDAP_TOKEN_TYPE, new LdapAuthorization());
    }

    public AuthorizeControllerVersion2(WebContext context) {
        super(context);
    }

    public static void registerAuthorizationDelegator(String name, AuthorizationDelegator delegator){
        delegatorMap.put(name, delegator);
    }

    @PostMapping(value = "/login", produces = "application/json", consumes = "application/json")
    @ResponseBody
    public AuthorizeResponse login(HttpServletRequest req, @RequestBody LoginPassword loginPassword){
        for (AuthorizationDelegator delegator :delegatorMap.values() ){
           if ( delegator.shouldAuthorize(req, getContext())){
               return delegator.authorize(req, loginPassword, getContext());
           }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }
    @PostMapping(value = "/request_login/{login}", produces = "application/json", consumes = "application/json")
    @ResponseBody
    public AuthorizeResponse requestLogin(HttpServletRequest req, @PathVariable String login){
        for (AuthorizationDelegator delegator :delegatorMap.values() ){
           if ( delegator.shouldAuthorize(req, getContext())){
               delegator.requestAuthorization(req,  getContext(), login);
               var res = new AuthorizeResponse();
               res.setSuccess(true);
               return res;
           }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }
    @PostMapping(value = "/request_registration/{login}", produces = "application/json", consumes = "application/json")
    @ResponseBody
    public AuthorizeResponse requestRegistration(HttpServletRequest req, @PathVariable String login){
        for (AuthorizationDelegator delegator :delegatorMap.values() ){
           if ( delegator.shouldAuthorize(req, getContext())){
               delegator.requestRegistration(req,  getContext(), login);
               var res = new AuthorizeResponse();
               res.setSuccess(true);
               return res;
           }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }

    @PostMapping(value = "/register", produces = "application/json", consumes = "application/json")
    public Boolean register(HttpServletRequest req, @RequestBody RegistrationForm loginPassword, @RequestParam(required = false) Long invitationId){
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(req, null, true);
        getBean(AuthScopeService.class).checkRegistrationAllowed(scope);

        for (AuthorizationDelegator delegator :delegatorMap.values() ){

           if ( delegator.shouldAuthorize(req, getContext())){
               return delegator.register( loginPassword, getContext(), scope, invitationId);
           }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }

    @WithRole("ROLE_ADMIN")
    @PostMapping("/block/{user}")
    public boolean blockUser(HttpServletRequest req, @PathVariable("user")  Long user){
        for (AuthorizationDelegator delegator :delegatorMap.values() ){
           if ( delegator.shouldAuthorize(req, getContext())){
               return delegator.blockUser(true, user, getContext() );
           }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }

    @GetMapping("/2factCode")
    public Auth2FactorEnabled get2FactCode(){
        var utils =getBean(SecurityUtils.class);
        return utils.get2factorBarCodeForUser(getBean(LoginPasswordService.class));
    }

    @PostMapping("/2factCode")
    public Auth2FactorEnabled set2FactEnabled(@RequestBody Auth2FactorEnabled request){
        var utils =getBean(SecurityUtils.class);

        var lp = getBean(LoginPasswordService.class).findUserByLogin(utils.get().getUserData().getLogin()).orElseThrow(() -> ItemNotFoundException.fromId(""));

        if (request.getEnabled()) {
            String code = utils.getTOTPCode(lp);
            if (!code.equalsIgnoreCase(request.getCode())) {
                throw new InvalidAuthenticatorCodeException("global.exceptions.invalid_code");
            }
        }

        Auth2FactorEnabled res = new Auth2FactorEnabled();
        res.setCode(lp.getTwoFactorCode());
        res.setEnabled(request.getEnabled());
        lp.setTwoFactor(request.getEnabled());
        //todo move to service
        getBean(LoginPasswordService.class).getRepository().save(lp);
        res.setEnabled(request.getEnabled());
        return res;
    }




    @PostMapping("/logout")
    public String logout(HttpServletRequest req){
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(req, getCurrentUser(), false);
        for (AuthorizationDelegator delegator :delegatorMap.values() ){
           if ( delegator.shouldAuthorize(req, getContext())){
               try {
                   return String.valueOf(delegator.logout(getBean(SecurityUtils.class).getTokenFromRequest().getToken(), getContext(), scope));
               } catch (AuthorizationNotFoundException exception){
                   throw new LogoutException("No token on request. cannot logout");
               }
           }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }

    @PostMapping("/password/reset")

    public String changePassword(HttpServletRequest req, @RequestBody ChangePasswordForm form){
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(req, getCurrentUser(), false);
        for (AuthorizationDelegator delegator :delegatorMap.values() ){
           if ( delegator.shouldAuthorize(req, getContext())){
               try {
                   return String.valueOf(delegator.changePassword(req, getContext(), form, scope));
               } catch (AuthorizationNotFoundException exception){
                   throw new ChangePasswordException("global.change_password.not_possible.wrong_auth", getContext().getBean(SecurityUtils.class).get().getUserData().getEmail());
               }
           }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }

    @PostMapping("/password/forgot/{login}")
    public void forgot(@PathVariable("login") String login){
        UserData userData = getContext().getBean(ProfilePageService.class)
                .findByLogin(login).orElseThrow(() -> ItemNotFoundException.fromId(login));
        VerificationTokenModel verificationToken = getBean(VerificationTokenService.class).saveToken(userData, 30);
        generateForgotAndSendForgotMessage(userData, verificationToken);
    }

    @PostMapping("/password/send_temporary")
    public String resetPassword(HttpServletRequest req, @RequestParam("login") String login, @RequestParam("token") String token) {
        getBean(VerificationTokenService.class).validateVerificationToken(token);

        for (AuthorizationDelegator delegator :delegatorMap.values() ){
            if (delegator.shouldAuthorize(req, getContext())){
                try {
                    String restore = String.valueOf(delegator.forgetPassword(req, getContext(), login));
                    getBean(VerificationTokenService.class).deleteToken(token);
                    return restore;
                } catch (AuthorizationNotFoundException exception){
                    throw new LogoutException("No token on request. cannot logout");
                }
            }
        }
        throw  new AuthorizationInvalidDelegator("auth type not supported : " + req.getHeader(AUTH_TYPE_HEADER) + " supported types: " + Arrays.toString(delegatorMap.keySet().toArray()));
    }

    @PostMapping("/verification/verify_scope")
    public Boolean verify(@RequestParam(value = "token") String token, @RequestParam(value = "scope") String scopeName) {
        getBean(VerificationTokenService.class).validateVerificationToken(token);
        VerificationTokenModel tokenModel = getBean(VerificationTokenService.class).getByToken(token);

        UserData userData = tokenModel.getUserData();
        if (Objects.nonNull(userData.getVerified()) && userData.getVerified()) {
            throw new VerificationException("global.exceptions.user_already_verified");
        }
        userData.setVerified(true);
        getBean(UserDataRepository.class).save(userData);

        tokenModel.setExpirationDate(new Date());
        getBean(VerificationTokenService.class).save(tokenModel);

        return true;
    }

    @PostMapping("/verification/resend")
    public Boolean resendVerification(@RequestParam(value = "email") String email, @RequestParam(value = "scope") String scopeName) {
        UserData userData = getBean(ProfilePageService.class).findByLogin(email).orElseThrow(() -> ItemNotFoundException.fromId(email));
        ScopeModel scope = getBean(AuthScopeService.class).findByName(scopeName, userData.getId(), false);

        if (!getBean(UserInScopeService.class).isRelationExist(userData, scope)) {
            throw new ScopeException("global.exception.user_not_in_scope");
        }

        getBean(VerificationTokenService.class).deleteActiveTokens(userData);
        getBean(UserInScopeService.class).sendVerificationEmail(scope, userData);
        return true;
    }

    private void generateForgotAndSendForgotMessage(UserData userData, VerificationTokenModel verificationToken) {
        ScopeModel scope = getBean(AuthScopeService.class).getScopeFromRequest(getContext().getRequest(), userData.getId(), false);
        String url = scope.getUrl() + getPasswordForgotPath() + "?login=" + userData.getLogin() + "&token=" + verificationToken.getToken();
        String path = userData.getPreferredLanguage() + "/forgot_password.html";

        Map<String, String> templateParameters = new HashMap<>();
        templateParameters.put("passwordLink", url);

        String content = getBean(MailSender.class).processHtmlTemplate(path, templateParameters);
        String title = getContext().getMessageTranslation("email.forgot_password.title", userData.getPreferredLanguage());

        getBean(MailSender.class).sendMail(Collections.singletonList(userData.getEmail()), content, title);
    }

    private String getBaseUrl() {
        return getContext().getEnv("app.settings.site.self.ui");
    }

    private String getPasswordForgotPath() {
        return getContext().getEnv("app.password.reset.path", "/forgot_password");
    }
}
