package systems.dennis.auth.client.required;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import systems.dennis.auth.client.utils.RoleValidationType;
import systems.dennis.auth.client.utils.SecurityUtils;
import systems.dennis.auth.role_validator.entity.UserTokenDTO;
import systems.dennis.auth.role_validator.exception.RolesNotFoundForTokenException;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.exceptions.AuthorizationNotFoundException;

import jakarta.servlet.http.HttpServletRequest;

import java.util.ArrayList;
import java.util.List;

import static systems.dennis.auth.role_validator.entity.UserRole.ROLE_SIGNED;

@Service
@Slf4j
public class TokenValidator {

    private final TokenProviderClient tokenProvider;
    private final SecurityUtils utils;


    public TokenValidator(TokenProviderClient tokenValidator, SecurityUtils utils) {
        this.tokenProvider = tokenValidator;
        this.utils = utils;
    }

    public void validate(HttpServletRequest request, List<String> roles, RoleValidationType validationType) throws AuthorizationNotFoundException, RolesNotFoundForTokenException {
        log.info("----- auth client started ------");
        if (request.getHeader("Authorization") == null) {
            if (utils.get() == null )
                throw new AuthorizationNotFoundException("No token, expected to have token");
        }

        if (roles != null && roles.isEmpty()) {
            return;
        }

        var token = utils.getTokenFromRequest();

        UserTokenDTO dto = tokenProvider.validateToken(token);

        if (dto == null) {
            throw new AuthorizationFailedException("Invalid token");
        }

        StringBuilder buffer = new StringBuilder();

        boolean foundOne = false;
        if (roles == null){
            return;
        }
        for (String role : roles) {
            if (role.equalsIgnoreCase(ROLE_SIGNED)){
                //we already check that the user is signed, nothing to do here
                continue;
            }
            boolean foundAll = false;
            for (String userRole : dto.getRoleList()) {
                if (role.equalsIgnoreCase(userRole)) {
                    if (validationType == RoleValidationType.ONE) {
                        foundOne = true;
                    }
                    foundAll = true;
                    break;
                }
            }
            if (!foundAll) {
                buffer.append("Role ").append(role).append(" not assigned to user");
            }
        }

        if (buffer.length() > 0 && validationType == RoleValidationType.ALL
                || !foundOne && validationType == RoleValidationType.ONE) {
            throw new RolesNotFoundForTokenException(buffer.toString());
        }
    }

    @SneakyThrows
    public List<String> getValidGroups(List<String> roles) {
        var token = utils.getTokenFromRequest();
        UserTokenDTO dto = tokenProvider.validateToken(token);

        List<String> validGroups = new ArrayList<>();
        if (dto != null && dto.getRoleList() != null) {
            validGroups = dto.getRoleList();
            if (roles != null) {
                validGroups.retainAll(roles);
            } else {
                validGroups.clear();
            }
        }

        if (roles != null && validGroups.size() == 0) {
            StringBuffer buffer = new StringBuffer();
            roles.forEach(role -> buffer.append("Role ").append(role).append(" not found "));
            log.warn(buffer.toString());
        }

        return validGroups;
    }


}
