package systems.dennis.shared.auth_client.beans;


import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import systems.dennis.shared.annotations.WebFormsSupport;
import systems.dennis.shared.annotations.security.*;
import systems.dennis.shared.annotations.security.selfchecker.AbstractSelfChecker;
import systems.dennis.shared.annotations.security.selfchecker.NoChecker;
import systems.dennis.shared.auth_client.SecurityUtils;
import systems.dennis.shared.auth_client.exception.RolesNotFoundForTokenException;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.controller.SearchEntityApi;
import systems.dennis.shared.entity.IDHolder;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.exceptions.AuthorizationNotFoundException;
import systems.dennis.shared.exceptions.ItemNotUserException;


import systems.dennis.shared.postgres.form.DefaultForm;
import systems.dennis.shared.postgres.model.LongAssignableEntity;
import systems.dennis.shared.service.AbstractService;
import systems.dennis.shared.utils.ApplicationContext;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;


/**
 * This class will add aop method to validate the token
 */
@Slf4j
@Aspect
public abstract class BasicAuthAoe extends ApplicationContext {


    private static final String ROLE_SIGNED = "ROLE_ANY";

    public BasicAuthAoe(WebContext context) {
        super(context);
        log.debug("Initializing Basic auth... ");
    }

    @Pointcut(value = "execution(public * *(..))")
    public void anyPublicMethod() {
    }

    @Pointcut(value = "@annotation(systems.dennis.shared.annotations.security.WithRole)")
    public void withRole2() {

    }

    // methods or classes with annotation @WithoutRole are excluded from Spring AOP
    @Before(value = "withRole2()")
    public void beforeAdvice(JoinPoint joinPoint) {

        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();

        if (method.getAnnotation(WithRole.class) == null) {
            return;
        }
        log.debug("In method: " + method.getName());

        Object[] values = joinPoint.getArgs();
        Object target = joinPoint.getTarget();

        if (checkPermission(method, target , values)) {
            return;
        }

        runAllSecurityAnnotations(method, values, target);
        checkSelf(method, target, values);

    }

    private boolean checkPermission(Method method, Object target, Object[] values) {
        Class<? extends PermissionCheck> cl = method.getAnnotation(WithRole.class).ignoreOnCondition();
        if (cl == NoCondition.class) {
            return false;
        }
        try {
            PermissionCheck instance = cl.getDeclaredConstructor().newInstance();

            if(target instanceof SearchEntityApi){
                String type = String.valueOf(values[0]);
                var correctTarget = getBean(SearchEntityApi.findServiceByType(type));
                return instance.checkIgnorePermission(values, correctTarget, getContext());
            } else {

                return instance.checkIgnorePermission(values, target, getContext());
            }
        } catch (Exception e) {
            log.error("Error during permission check: " + e.getMessage());
            return false;
        }
    }

    @SneakyThrows
    private void checkSelf(Method method, Object target, Object[] values) {
        var utils = getContext().getBean(SecurityUtils.class);
        if (method.getAnnotation(SelfOnlyRole.class) == null) {
            return;
        }
        var ann = target.getClass().getAnnotation(WebFormsSupport.class);

        if (ann == null) {
            throw new IllegalArgumentException("Self check can be applied to object that contains interface RepoService ");
        }
        //we assume that id parameter is always first
        //todo are we able to check with @Annotation?
        var parameters = method.getParameters();
        if (parameters == null || parameters.length == 0) {
            throw new IllegalArgumentException("Self role without ID detection, very strange....");
        }

        int arg = -1;

        if (parameters.length == 1) {
            arg = 0;
        }

        Id idAnnotation = null;
        for (int i = 0, parametersLength = parameters.length; i < parametersLength; i++) {
            Parameter p = parameters[i];
            if (p.getAnnotation(Id.class) != null) {
                arg = i;
                idAnnotation = p.getAnnotation(Id.class);
                break;
            }
        }

        if (arg == -1) {
            throw new IllegalArgumentException("NO @ID for the argument, haven't your forgot to add one? " + method);
        }

        AbstractService bean = getBean(ann.value());
        IDHolder entity = null;


        if (values[arg] instanceof DefaultForm) {
            entity = (DefaultForm) values[arg];

            try {
                var isUsers = bean.getByIdAndUserDataId(entity.getId(), getCurrentUser());
                if (!isUsers) {
                   bean.checkMy(bean.findByIdOrThrow(entity.getId()));
                }
            } catch (Exception e) {
                // Not User Entity or entity doesn't have property userDataId
                throw new ItemNotUserException();
            }
            return;
        } else {
            entity = bean.findByIdOrThrow((Long) values[arg]);
        }

        if (!(entity instanceof LongAssignableEntity)) {
            if (idAnnotation != null) {
                if (idAnnotation.checker() == NoChecker.class) {
                    // throw new IllegalArgumentException("Entity of class: " + entity.getClass() + " is not really instance of LongAssignableEntity");
                    log.info("Entity of class: " + entity.getClass() + " is not really instance of LongAssignableEntity");
                } else {
                    AbstractSelfChecker checker = idAnnotation.checker().getConstructor().newInstance();
                    checker.check(utils.tokenFromHeader(), entity, getContext());
                }
            }
        } else {
            utils.isMy(entity);
        }


    }

    @SneakyThrows
    private void runAllSecurityAnnotations(Method method, Object[] values, Object target) {
        HttpServletRequest request = getContext().getRequest();

        if (request == null) {
            log.error("No HttpServletRequest");
            log.error("HttpServletRequest is not existing on " + target.getClass().getName());
            return;
        }


        List<String> requestedRoles = new ArrayList<>();
        requestedRoles.addAll(getRolesByAnnotation(method, request));
        requestedRoles.addAll(additionalRoles(target));
        validate(request, requestedRoles, RoleValidationType.ALL);
    }


    @SneakyThrows
    private List<String> getRolesByAnnotation(Method method, HttpServletRequest request) {
        WithRole withRole = method.getAnnotation(WithRole.class);
        if (withRole != null) {
            if (!withRole.or().equals("")) {
                try {
                    validate(request, Collections.singletonList(withRole.value()), RoleValidationType.ALL);
                } catch (Exception e) {
                    return Collections.singletonList(withRole.or());
                }
            } else {
                return Collections.singletonList(withRole.value());
            }
        }
        WithRoles withRoles = method.getAnnotation(WithRoles.class);
        if (withRoles == null) {
            return Collections.emptyList();
        }
        List<String> result = new ArrayList<>();
        for (WithRole role : withRoles.roles()) {
            result.add(role.value());
        }
        return result;
    }


    public List<String> additionalRoles(Object target) {
        if (target.getClass().getAnnotation(Secured.class) != null) {

            return Arrays.asList(target.getClass().getAnnotation(Secured.class).roles());
        } else {
            return Collections.emptyList();
        }
    }

    public void validate(HttpServletRequest request, List<String> roles, RoleValidationType validationType) throws AuthorizationNotFoundException, RolesNotFoundForTokenException {
        log.info("----- auth client started ------");
        var token = getBean(SecurityUtils.class).get();
        if (request.getHeader("Authorization") == null) {
            if (token == null )
                throw new AuthorizationNotFoundException("No token, expected to have token");
        }

        if (roles != null && roles.isEmpty()) {
            return;
        }


        if (token == null) {
            throw new AuthorizationFailedException("Invalid token");
        }

        token.validate(getContext());

        StringBuilder buffer = new StringBuilder();

        boolean foundOne = false;
        if (roles == null){
            return;
        }
        for (String role : roles) {
            if (role.equalsIgnoreCase(ROLE_SIGNED)){
                //we already check that the user is signed, nothing to do here
                continue;
            }
            boolean foundAll = false;
            for (String userRole : token.getRoleList()) {
                if (role.equalsIgnoreCase(userRole)) {
                    if (validationType == RoleValidationType.ONE) {
                        foundOne = true;
                    }
                    foundAll = true;
                    break;
                }
            }
            if (!foundAll) {
                buffer.append("Role ").append(role).append(" not assigned to user");
            }
        }

        if (buffer.length() > 0 && validationType == RoleValidationType.ALL
                || !foundOne && validationType == RoleValidationType.ONE) {
            throw new RolesNotFoundForTokenException(buffer.toString());
        }
    }

}
