package systems.dennis.shared.auth_client;

import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;
import systems.dennis.shared.annotations.security.ISecurityUtils;
import systems.dennis.shared.auth_client.client.TokenProviderClient;
import systems.dennis.shared.auth_client.form.UserTokenDTO;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.entity.AbstractEntity;
import systems.dennis.shared.entity.TokenData;
import systems.dennis.shared.exceptions.AuthorizationNotFoundException;
import systems.dennis.shared.model.AbstractUserAssignableElement;
import systems.dennis.shared.utils.ApplicationContext;

import java.io.Serializable;

@Service
public class SecurityUtils<T extends Serializable> extends ApplicationContext implements ISecurityUtils<T> {
    private HttpServletRequest request;
    public static final String AUTHORIZATION_HEADER = "Authorization";
    public static String AUTH_TYPE_HEADER = "AUTH_TYPE";
    public static String AUTH_SCOPE_HEADER = "AUTH_SCOPE";
    public SecurityUtils(WebContext context, HttpServletRequest request) {
        super(context);
        this.request = request;
    }

    @Override
    public  T getUserDataId() {
        return (T) tokenFromHeader().getUserData().getId();
    }

    @SneakyThrows
    public UserTokenDTO tokenFromHeader() throws AuthorizationNotFoundException {

        return getBean(TokenProviderClient.class).getAuthentication(getToken());

    }


    public TokenData getToken() {

        if (request == null) {
            return null;
        }
        String token = request.getHeader(AUTHORIZATION_HEADER);
        String tokenScope = request.getHeader(AUTH_SCOPE_HEADER);

        if (token == null) {
            throw  new AuthorizationNotFoundException("Expected token, instead no token");

        }

        token = token.replace("Bearer ", "").trim();
        return new TokenData( tokenScope,token);
    }



    @Override
    public void assignUser(AbstractEntity<T> pojo) {
        if (pojo instanceof AbstractUserAssignableElement) {
            ((AbstractUserAssignableElement) pojo).setUserDataId(getCurrentUser());
        }
    }

    @Override
    public void assignUser(AbstractEntity<T> pojo, AbstractEntity<T> pojoOriginal) {
        if (pojo instanceof AbstractUserAssignableElement) {
            ((AbstractUserAssignableElement) pojo).setUserDataId(((AbstractUserAssignableElement) pojoOriginal).getUserDataId());
        }
    }


    @Override
    public boolean isAdmin() {
        try {
            return hasRole("ROLE_ADMIN");
        } catch (Exception e) {
            return false;
        }

    }

    @Override
    public boolean hasRole(String role) {
        return  tokenFromHeader().getRoleList().contains(role);
    }

    @Override
    public String getUserLanguage() {
        return tokenFromHeader().getUserData().getPreferredLanguage();
    }

    public UserTokenDTO get() {
        return tokenFromHeader();
    }
}
