package systems.dennis.shared.auth_client;

import jakarta.servlet.http.HttpServletRequest;
import lombok.SneakyThrows;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import systems.dennis.shared.annotations.security.ISecurityUtils;
import systems.dennis.shared.auth_client.client.TokenProviderClient;
import systems.dennis.shared.auth_client.form.UserTokenDTO;
import systems.dennis.shared.beans.IdToAuthorizationIdBean;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.entity.AbstractEntity;
import systems.dennis.shared.entity.TokenData;
import systems.dennis.shared.exceptions.AuthorizationNotFoundException;

import systems.dennis.shared.utils.ApplicationContext;

import java.io.Serializable;

@Service @Primary
public class SecurityUtils<T extends Serializable> extends ApplicationContext implements ISecurityUtils<T> {
    private HttpServletRequest request;
    private final IdToAuthorizationIdBean id2id;
    public static final String AUTHORIZATION_HEADER = "Authorization";
    String AUTH_TYPE_HEADER = "AUTH_TYPE";
    String AUTH_SCOPE_HEADER = "AUTH_SCOPE";
    public SecurityUtils(WebContext context, HttpServletRequest request, IdToAuthorizationIdBean id2id) {
        super(context);
        this.request = request;
        this.id2id = id2id;
    }

    @Override
    public  T getUserDataId() {
        return (T) id2id.idToAuthorizationId( tokenFromHeader().getUserData().getId());
    }

    @SneakyThrows
    public UserTokenDTO tokenFromHeader() throws AuthorizationNotFoundException {

        return getBean(TokenProviderClient.class).getAuthentication(getToken());

    }


    public TokenData getToken() {

        if (request == null) {
            return null;
        }
        String token = request.getHeader(AUTHORIZATION_HEADER);
        String tokenScope = request.getHeader(AUTH_SCOPE_HEADER);

        if (token == null) {
            throw  new AuthorizationNotFoundException("Expected token, instead no token");

        }

        token = token.replace("Bearer ", "").trim();
        return new TokenData( tokenScope,token);
    }


    @Override
    public boolean isAdmin() {
        return hasRole("ROLE_ADMIN");
    }

    @Override
    public boolean hasRole(String role) {
        return  tokenFromHeader().getRoleList().contains(role);
    }

    @Override
    public String getUserLanguage() {
        return tokenFromHeader().getUserData().getPreferredLanguage();
    }

    public UserTokenDTO get() {
        return tokenFromHeader();
    }
}
