package systems.dennis.shared.auth_client.client;

import lombok.SneakyThrows;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.stereotype.Service;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import systems.dennis.shared.auth_client.controller.AuthorizeResponse;
import systems.dennis.shared.auth_client.exception.NeedAuthorizationException;
import systems.dennis.shared.auth_client.form.LoginPassword;
import systems.dennis.shared.auth_client.form.UserTokenDTO;
import systems.dennis.shared.config.WebContext;
import systems.dennis.shared.exceptions.AccessDeniedException;
import systems.dennis.shared.exceptions.AuthorizationFailedException;
import systems.dennis.shared.exceptions.StandardException;
import systems.dennis.shared.servers.beans.ServerRequest;
import systems.dennis.shared.servers.model.ServerConfig;
import systems.dennis.shared.servers.providers.ServerTypeProvider;
import systems.dennis.shared.servers.service.ServerConfigService;

@Service
@ConditionalOnExpression("${global.auth.client.enabled:false}")
public class AuthClient {

    private final RestTemplate restTemplate;
    private final WebContext context;

    public AuthClient(RestTemplate restTemplate, WebContext context) {
        this.restTemplate = restTemplate;
        this.context = context;

    }


    @SneakyThrows
    public String login() {
        var res = getVirtualToken();
        if (res == null) {
            throw new AccessDeniedException("user_login_password_is_incorrect");
        } else {
            return res.getToken();
        }
    }

    @SneakyThrows
    public String login(String userName, String userPassword, String authCode) {
        var res = getToken(userName, userPassword, authCode);
        if (res == null) {
            throw new NeedAuthorizationException("auth.client.authorization.failed");
        } else {
            return res.getToken();
        }

    }

    public UserTokenDTO getVirtualToken() {
        try {
            LoginPassword param = new LoginPassword();
            param.setPassword(getServerPassword());
            param.setLogin(getServerLogin());



            return new ServerRequest<AuthorizeResponse>(restTemplate, context).virtualAuth()
                    .server(ServerTypeProvider.VIRTUAL_AUTH).scope()
                    .uri("api/v3/auth/login")
                    .onErrorStatusError(403, (obj) -> {
                        throw new AccessDeniedException("user_has_no_permission_for this");
                    })
                    .onErrorStatusError(401, (obj) -> {
                        throw new AuthorizationFailedException("user_login_password_is_incorrect");
                    })
                    .onAnyErrorStatusError((obj) -> {
                        throw new StandardException("not_able_to_get_result_status", obj + "");
                    })
                    .executePost(param, AuthorizeResponse.class).getDto();

        } catch (HttpClientErrorException.Forbidden e) {
            throw new AccessDeniedException("user_login_password_is_incorrect");
        }
    }

    @SneakyThrows
    private UserTokenDTO getToken(String login, String password, String authCode) {
        try {
            LoginPassword param = new LoginPassword();
            param.setPassword(password);
            param.setLogin(login);
            param.setTwoFactorCode(authCode);
            return new ServerRequest<AuthorizeResponse>(restTemplate, context).defaultAuthType()
                    .server(ServerTypeProvider.VIRTUAL_AUTH).scope()
                    .uri("api/v3/auth/login")
                    .onErrorStatusError(403, (obj) -> {
                        throw new AccessDeniedException("user_has_no_permission_for this");
                    })
                    .onErrorStatusError(401, (obj) -> {
                        throw new AuthorizationFailedException("user_login_password_is_incorrect");
                    })
                    .onAnyErrorStatusError((obj) -> {
                        throw new StandardException("not_able_to_get_result_status", obj + "");
                    })
                    .executePost(param, AuthorizeResponse.class).getDto();
        } catch (HttpClientErrorException.Forbidden e) {
            throw new AccessDeniedException("user_login_password_is_incorrect");
        }
    }

    private String getServerLogin() {

        var server = findServer(ServerTypeProvider.VIRTUAL_AUTH);

        if (server != null) {
            return server.getUserName();
        }

        return context.getEnv("servers.central.user");
    }

    private ServerConfig findServer(Long type) {
        return context.getBean(ServerConfigService.class).findByType(type, true);
    }

    private String getServerPassword() {
        var server = context.getBean(ServerConfigService.class).findByType(ServerTypeProvider.VIRTUAL_AUTH, true);
        if (server != null) {
            return server.getPassword();
        }

        return context.getEnv("servers.central.password");
    }

    private String getRegisterLink() {
        return context.getEnv("pages.auth.register_link", "/register/add");
    }

    private String getRestoreLink() {
        return context.getEnv("pages.auth.restore_link", "/auth/profile/restore");
    }

    private String getAuthPath() {
        var server = context.getBean(ServerConfigService.class).findByType(ServerTypeProvider.VIRTUAL_AUTH, true);
        if (server != null) {
            return server.getHost() + "/" + context.getEnv("servers.auth.request_path", "api/v3/auth");
        }
        return context.getEnv("servers.auth.path");
    }

    private String getScopeId() {
        return context.getEnv("dennis.systems.security.scope.id").toString();
    }
}
