/*
 * Decompiled with CFR 0.152.
 */
package com.pusher.client.connection.websocket;

import com.pusher.client.channel.impl.ChannelManager;
import com.pusher.client.connection.ConnectionEventListener;
import com.pusher.client.connection.ConnectionState;
import com.pusher.client.connection.ConnectionStateChange;
import com.pusher.client.connection.impl.InternalConnectionManager;
import com.pusher.client.util.PusherJsonParser;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import jdk.incubator.http.HttpClient;
import jdk.incubator.http.WebSocket;

public final class WebSocketConnectionManager
implements InternalConnectionManager,
WebSocket.Listener {
    private static final System.Logger log = System.getLogger(WebSocketConnectionManager.class.getName());
    private static final String INTERNAL_EVENT_PREFIX = "pusher:";
    private static final byte[] PING_MSG = "ping".getBytes(StandardCharsets.UTF_8);
    private final HttpClient httpClient;
    private final Executor listenerExecutor;
    private final ScheduledExecutorService timers;
    private final ChannelManager channelManager;
    private final Map<ConnectionState, Set<ConnectionEventListener>> eventListeners;
    private final URI webSocketUri;
    private final int maxReconnectionGap;
    private final long pongTimeout;
    private volatile ConnectionState state = ConnectionState.DISCONNECTED;
    private volatile CompletableFuture<WebSocket> webSocket;
    private volatile StringBuilder partialMsgBuilder;
    private volatile String socketId;
    private volatile int reconnectAttempts;
    private volatile long activityTimeout;
    private volatile ScheduledFuture<?> connectFuture;
    private volatile long lastActivity;
    private volatile long lastPing;
    private volatile ScheduledFuture<?> activityMonitorFuture;

    public WebSocketConnectionManager(String name, URI webSocketUri, HttpClient httpClient, ChannelManager channelManager, long activityTimeout, long pongTimeout, int maxReconnectionGap, Executor listenerExecutor) {
        this.webSocketUri = webSocketUri;
        this.httpClient = httpClient == null ? HttpClient.newHttpClient() : httpClient;
        this.channelManager = channelManager;
        this.activityTimeout = activityTimeout;
        this.pongTimeout = pongTimeout;
        this.eventListeners = new ConcurrentHashMap<ConnectionState, Set<ConnectionEventListener>>();
        this.maxReconnectionGap = maxReconnectionGap;
        this.listenerExecutor = listenerExecutor;
        this.timers = Executors.newScheduledThreadPool(1, new PusherThreadFactory(name));
    }

    @Override
    public void connect() {
        if (this.state == ConnectionState.DISCONNECTED) {
            this.tryConnecting();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tryConnecting() {
        ScheduledExecutorService scheduledExecutorService = this.timers;
        synchronized (scheduledExecutorService) {
            try {
                switch (this.state) {
                    case RECONNECTING: 
                    case DISCONNECTED: {
                        break;
                    }
                    default: {
                        return;
                    }
                }
                this.updateState(ConnectionState.CONNECTING);
                this.webSocket = this.httpClient.newWebSocketBuilder().connectTimeout(Duration.ofMillis(this.pongTimeout)).buildAsync(this.webSocketUri, (WebSocket.Listener)this);
            }
            catch (RuntimeException e) {
                this.sendErrorToAllListeners("Error connecting to WebSocket server.", null, e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void disconnect() {
        ScheduledExecutorService scheduledExecutorService = this.timers;
        synchronized (scheduledExecutorService) {
            if (this.state == ConnectionState.CONNECTED) {
                this.webSocket = this.webSocket.join().sendClose(1000, "");
                this.updateState(ConnectionState.DISCONNECTING);
            }
        }
    }

    @Override
    public void bind(ConnectionState state, ConnectionEventListener eventListener) {
        this.eventListeners.computeIfAbsent(state, _state -> Collections.newSetFromMap(new ConcurrentHashMap())).add(eventListener);
    }

    @Override
    public boolean unbind(ConnectionState state, ConnectionEventListener eventListener) {
        Set<ConnectionEventListener> listeners = this.eventListeners.get((Object)state);
        return listeners != null && listeners.remove(eventListener);
    }

    @Override
    public ConnectionState getState() {
        return this.state;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void sendMessage(String message) {
        ScheduledExecutorService scheduledExecutorService = this.timers;
        synchronized (scheduledExecutorService) {
            try {
                this.webSocket = this.webSocket.join().sendText((CharSequence)message, true);
            }
            catch (Exception e) {
                this.sendErrorToAllListeners("An exception occurred while sending message [" + message + "]", null, e);
            }
        }
    }

    @Override
    public String getSocketId() {
        return this.socketId;
    }

    private void updateState(ConnectionState newState) {
        Set<ConnectionEventListener> newStateListeners;
        log.log(System.Logger.Level.TRACE, "State transition requested, current [" + this.state + "], new [" + newState + "]");
        ConnectionStateChange change = new ConnectionStateChange(this.state, newState);
        this.state = newState;
        HashSet dedupe = new HashSet();
        Set<ConnectionEventListener> allStateListeners = this.eventListeners.get((Object)ConnectionState.ALL);
        if (allStateListeners != null) {
            allStateListeners.stream().filter(dedupe::add).forEach(listener -> this.listenerExecutor.execute(() -> listener.onConnectionStateChange(change)));
        }
        if ((newStateListeners = this.eventListeners.get((Object)newState)) != null) {
            newStateListeners.stream().filter(dedupe::add).forEach(listener -> this.listenerExecutor.execute(() -> listener.onConnectionStateChange(change)));
        }
    }

    private void handleEvent(String rawJson) {
        String event = PusherJsonParser.getJsonValue(rawJson, "\"event\":");
        if (event == null) {
            throw new IllegalArgumentException("Message does not contain an event field: " + rawJson);
        }
        if (event.startsWith(INTERNAL_EVENT_PREFIX)) {
            this.handleInternalEvent(event, rawJson);
            return;
        }
        this.channelManager.onMessage(event, rawJson);
    }

    private void handleInternalEvent(String event, String rawJson) {
        if (event.equals("pusher:connection_established")) {
            this.handleConnectionMessage(rawJson);
        } else if (event.equals("pusher:error")) {
            this.handleError(rawJson);
        }
    }

    private void handleError(String rawJson) {
        String message = PusherJsonParser.getJsonValue(rawJson, "\"message\":");
        String code = PusherJsonParser.getJsonValue(rawJson, "\"code\":");
        this.sendErrorToAllListeners(message, code, null);
    }

    private void sendErrorToAllListeners(String message, String code, Throwable e) {
        HashSet dedupe = new HashSet();
        this.eventListeners.values().stream().flatMap(Collection::stream).filter(dedupe::add).forEach(listener -> this.listenerExecutor.execute(() -> listener.onError(message, code, e)));
    }

    public void onOpen(WebSocket webSocket) {
        webSocket.request(Long.MAX_VALUE);
    }

    public CompletionStage<?> onPing(WebSocket webSocket, ByteBuffer message) {
        this.lastActivity = System.currentTimeMillis();
        return webSocket.sendPong(message);
    }

    public CompletionStage<?> onPong(WebSocket webSocket, ByteBuffer message) {
        this.lastActivity = System.currentTimeMillis();
        return null;
    }

    public CompletionStage<?> onText(WebSocket webSocket, CharSequence message, WebSocket.MessagePart part) {
        switch (part) {
            case WHOLE: {
                String msgCopy = message.toString();
                this.listenerExecutor.execute(() -> this.handleEvent(msgCopy));
                return null;
            }
            case FIRST: {
                if (this.partialMsgBuilder == null) {
                    this.partialMsgBuilder = new StringBuilder(message);
                    return null;
                }
                this.partialMsgBuilder.setLength(0);
                this.partialMsgBuilder.append(message);
                return null;
            }
            case PART: {
                this.partialMsgBuilder.append(message);
                return null;
            }
            case LAST: {
                String fullMessage = this.partialMsgBuilder.append(message).toString();
                this.listenerExecutor.execute(() -> this.handleEvent(fullMessage));
                return null;
            }
        }
        throw new IllegalStateException(part + " WebSocket messages are not currently supported. Please open an issue with replication details.");
    }

    public CompletionStage<?> onClose(WebSocket webSocket, int statusCode, String reason) {
        switch (this.state) {
            case RECONNECTING: 
            case DISCONNECTED: {
                log.log(System.Logger.Level.WARNING, "Received close from underlying socket when already disconnected.  Close code [" + statusCode + "], Reason [" + reason + "]");
                break;
            }
            case CONNECTING: 
            case CONNECTED: {
                this.tryReconnecting();
                break;
            }
            case DISCONNECTING: {
                this.cancelTimeoutsAndTransitionToDisconnected();
                break;
            }
        }
        this.partialMsgBuilder = null;
        return null;
    }

    public void onError(WebSocket webSocket, Throwable error) {
        this.sendErrorToAllListeners("An exception was thrown by the WebSocket.", null, error);
        switch (this.state) {
            case CONNECTING: 
            case CONNECTED: {
                this.tryReconnecting();
                break;
            }
            case DISCONNECTING: {
                this.cancelTimeoutsAndTransitionToDisconnected();
                break;
            }
        }
        this.partialMsgBuilder = null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tryReconnecting() {
        ScheduledExecutorService scheduledExecutorService = this.timers;
        synchronized (scheduledExecutorService) {
            if (this.connectFuture != null && !this.connectFuture.isDone()) {
                return;
            }
            ++this.reconnectAttempts;
            this.updateState(ConnectionState.RECONNECTING);
            long reconnectInterval = Math.min(this.maxReconnectionGap, Math.abs(this.reconnectAttempts * this.reconnectAttempts));
            this.connectFuture = this.timers.schedule(this::tryConnecting, reconnectInterval, TimeUnit.SECONDS);
        }
    }

    private void cancelTimeoutsAndTransitionToDisconnected() {
        if (this.activityMonitorFuture != null) {
            this.activityMonitorFuture.cancel(false);
        }
        this.updateState(ConnectionState.DISCONNECTED);
    }

    private void handleConnectionMessage(String rawJson) {
        String activityTimeoutString;
        this.socketId = PusherJsonParser.getEscapedJsonValue(rawJson, "\\\"socket_id\\\":");
        if (this.state != ConnectionState.CONNECTED) {
            this.updateState(ConnectionState.CONNECTED);
        }
        if ((activityTimeoutString = PusherJsonParser.getEscapedJsonValue(rawJson, "\\\"activity_timeout\\\":")) != null) {
            this.activityTimeout = Math.min(this.activityTimeout, 1000L * Long.parseLong(activityTimeoutString));
        }
        this.lastActivity = System.currentTimeMillis();
        this.scheduleActivityCheck();
        this.reconnectAttempts = 0;
    }

    private void scheduleActivityCheck() {
        this.activityMonitorFuture = this.timers.scheduleWithFixedDelay(() -> {
            long now = System.currentTimeMillis();
            if (now - this.lastActivity < this.activityTimeout) {
                return;
            }
            if (this.lastPing > this.lastActivity && now - this.lastPing > this.pongTimeout) {
                log.log(System.Logger.Level.TRACE, "Timed out awaiting pong from server - disconnecting");
                this.disconnect();
                this.onClose(null, -1, "Pong timeout");
                return;
            }
            ScheduledExecutorService scheduledExecutorService = this.timers;
            synchronized (scheduledExecutorService) {
                switch (this.state) {
                    case CONNECTING: {
                        this.tryReconnecting();
                        return;
                    }
                    case CONNECTED: {
                        log.log(System.Logger.Level.TRACE, "Sending ping");
                        this.webSocket = this.webSocket.join().sendPing(ByteBuffer.wrap(PING_MSG)).whenComplete((ws, ex) -> {
                            if (ex == null) {
                                this.lastPing = System.currentTimeMillis();
                            }
                        });
                        return;
                    }
                }
            }
        }, this.activityTimeout, this.pongTimeout / 2L, TimeUnit.MILLISECONDS);
    }

    private static final class PusherThreadFactory
    implements ThreadFactory {
        private final String namePrefix;
        private final AtomicInteger nextId = new AtomicInteger(1);

        private PusherThreadFactory(String name) {
            this.namePrefix = name + "-pusher-java-client-connection-monitor-";
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(null, r, this.namePrefix + this.nextId.getAndIncrement(), 0L, false);
            t.setDaemon(true);
            return t;
        }
    }
}

