package com.pusher.client.channel.impl;

import com.pusher.client.AuthorizationFailureException;
import com.pusher.client.Authorizer;
import com.pusher.client.channel.*;
import com.pusher.client.connection.ConnectionState;
import com.pusher.client.connection.impl.InternalConnectionManager;

import java.util.concurrent.Executor;

public class PrivateChannelImpl<L extends PrivateChannelEventListener> extends ChannelImpl<L> implements PrivateChannel {

  private static final String CLIENT_EVENT_PREFIX = "client-";

  private final InternalConnectionManager connection;
  private final Authorizer authorizer;

  public PrivateChannelImpl(final InternalConnectionManager connection,
                            final String channelName,
                            final Authorizer authorizer,
                            final Executor listenerExecutor) {
    super(channelName, listenerExecutor);
    this.connection = connection;
    this.authorizer = authorizer;
  }

  /* PrivateChannel implementation */

  @Override
  @SuppressWarnings("rawtypes")
  public void trigger(final String eventName, final String data) {
    if (eventName == null || !eventName.startsWith(CLIENT_EVENT_PREFIX)) {
      throw new IllegalArgumentException("Cannot trigger event " + eventName
          + ": client events must start with \"client-\"");
    }
    if (state != ChannelState.SUBSCRIBED) {
      throw new IllegalStateException("Cannot trigger event " + eventName + " because channel " + channelName
          + " is in " + state.toString() + " state");
    }
    if (connection.getState() != ConnectionState.CONNECTED) {
      throw new IllegalStateException("Cannot trigger event " + eventName + " because connection is in "
          + connection.getState().toString() + " state");
    }
    connection.sendMessage("{\"data\":\"" + data.replace("\"", "\\\"")
        + "\",\"event\":\"" + eventName
        + "\",\"channel\":\"" + channelName + "\"}");
  }

  /* Base class overrides */

  @Override
  @SuppressWarnings("rawtypes")
  public String toSubscribeMessage() {
    final var authResponse = getAuthResponse();
    try {
      return "{\"event\":\"pusher:subscribe\",\"data\":{\"channel\":\"" + channelName + "\"," + authResponse.substring(1) + "}";
    } catch (final Exception e) {
      throw new AuthorizationFailureException("Unable to parse response from Authorizer: " + authResponse, e);
    }
  }

  @Override
  protected String[] getDisallowedNameExpressions() {
    return new String[]{"^(?!private-).*"};
  }

  /**
   * Protected access because this is also used by PresenceChannelImpl.
   */
  String getAuthResponse() {
    return authorizer.authorize(getChannelName(), connection.getSocketId());
  }

  @Override
  public String toString() {
    return String.format("[Private Channel: channelName=%s]", channelName);
  }
}
