package com.pusher.client;

import com.pusher.client.channel.impl.ChannelManager;
import com.pusher.client.connection.websocket.WebSocketConnectionManager;
import jdk.incubator.http.HttpClient;

import java.net.URI;
import java.util.Properties;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;

/**
 * Configuration for a {@link com.pusher.client.Pusher} instance.
 */
public final class PusherBuilder {

  private static final String SRC_LIB_DEV_VERSION = "@version@";
  private static final String LIB_DEV_VERSION = "0.0.0-dev";
  public static final String LIB_VERSION = readVersionFromProperties();

  private static final String URI_PARAMS = "?client=comodal&protocol=7&version=" + LIB_VERSION;
  private static final String WS_SCHEME = "ws";
  private static final String WSS_SCHEME = "wss";

  private static final int WS_PORT = 80;
  private static final int WSS_PORT = 443;
  // Note that the primary cluster lives on a different domain
  // (others are subdomains of pusher.com). This is not an oversight.
  // Legacy reasons.
  private static final String DEFAULT_HOST = "ws.pusherapp.com";
  private static final String PUSHER_DOMAIN = "pusher.com";

  private static final long DEFAULT_ACTIVITY_TIMEOUT = 14_000;
  private static final long DEFAULT_PONG_TIMEOUT = 7_000;

  private static final int MAX_RECONNECT_GAP_IN_SECONDS = 30;

  private String cluster = null;
  private boolean encrypted = true;
  private long activityTimeout = DEFAULT_ACTIVITY_TIMEOUT;
  private long pongTimeout = DEFAULT_PONG_TIMEOUT;
  private Authorizer authorizer;
  private int maxReconnectGapInSeconds = MAX_RECONNECT_GAP_IN_SECONDS;
  private HttpClient httpClient;
  private Executor listenerExecutor;

  PusherBuilder() {
  }

  public Pusher create(final String apiKey) {
    return create("my", apiKey);
  }

  public Pusher create(final String name, final String apiKey) {
    if (apiKey == null || apiKey.length() == 0) {
      throw new IllegalArgumentException("API Key cannot be null or empty");
    }
    return create(name, createURI(encrypted, cluster, apiKey));
  }

  public Pusher create(final String name, final URI uri) {
    final var listenerExecutor = this.listenerExecutor == null
        ? ForkJoinPool.commonPool()
        : this.listenerExecutor;
    final var channelManager = new ChannelManager(listenerExecutor);
    final var connectionManager = new WebSocketConnectionManager(
        name,
        uri,
        httpClient,
        channelManager,
        activityTimeout,
        pongTimeout,
        maxReconnectGapInSeconds,
        listenerExecutor);
    channelManager.setConnection(connectionManager);
    return new Pusher(listenerExecutor, channelManager, connectionManager, authorizer);
  }

  public Executor getListenerExecutor() {
    return listenerExecutor;
  }

  public PusherBuilder setListenerExecutor(final Executor listenerExecutor) {
    this.listenerExecutor = listenerExecutor;
    return this;
  }

  public HttpClient getHttpClient() {
    return httpClient;
  }

  public PusherBuilder setHttpClient(final HttpClient httpClient) {
    this.httpClient = httpClient;
    return this;
  }

  /**
   * Gets whether an encrypted (SSL) connection should be used when connecting
   * to Pusher.
   *
   * @return true if an encrypted connection should be used; otherwise false.
   */
  public boolean isEncrypted() {
    return encrypted;
  }

  /**
   * Sets whether an encrypted (SSL) connection should be used when connecting to
   * Pusher.
   *
   * @param encrypted Whether to use an SSL connection
   * @return this, for chaining
   */
  public PusherBuilder setEncrypted(final boolean encrypted) {
    this.encrypted = encrypted;
    return this;
  }

  /**
   * Gets the authorizer to be used when authenticating private and presence
   * channels.
   *
   * @return the authorizer
   */
  public Authorizer getAuthorizer() {
    return authorizer;
  }

  /**
   * Sets the authorizer to be used when authenticating private and presence
   * channels.
   *
   * @param authorizer The authorizer to be used.
   * @return this, for chaining
   */
  public PusherBuilder setAuthorizer(final Authorizer authorizer) {
    this.authorizer = authorizer;
    return this;
  }


  public PusherBuilder setCluster(final String cluster) {
    this.cluster = cluster;
    return this;
  }

  public String getCluster() {
    return cluster;
  }

  /**
   * The number of milliseconds of inactivity at which a "ping" will be
   * triggered to check the connection.
   * <p>
   * The default value is 14,000. On some connections, where
   * intermediate hops between the application and Pusher are aggressively
   * culling connections they consider to be idle, a lower value may help
   * preserve the connection.
   *
   * @param activityTimeout time to consider connection idle, in milliseconds
   * @return this, for chaining
   */
  public PusherBuilder setActivityTimeout(final long activityTimeout) {
    if (activityTimeout < 1000) {
      throw new IllegalArgumentException(
          "Activity timeout must be at least 1,000ms (and is recommended to be much higher)");
    }
    this.activityTimeout = activityTimeout;
    return this;
  }

  public long getActivityTimeout() {
    return activityTimeout;
  }

  /**
   * The number of milliseconds after a "ping" is sent that the client will
   * wait to receive a "pong" response from the server before considering the
   * connection broken and triggering a transition to the disconnected state.
   * <p>
   * The default value is 7,000.
   *
   * @param pongTimeout time to wait for pong response, in milliseconds
   * @return this, for chaining
   */
  public PusherBuilder setPongTimeout(final long pongTimeout) {
    if (pongTimeout < 1000) {
      throw new IllegalArgumentException(
          "Pong timeout must be at least 1,000ms (and is recommended to be much higher)");
    }
    this.pongTimeout = pongTimeout;
    return this;
  }

  /**
   * The delay in two reconnection extends exponentially (1, 2, 4, .. seconds) This property sets the maximum in between two
   * reconnection attempts.
   *
   * @param maxReconnectGapInSeconds time in seconds of the maximum gab between two reconnection attempts, default = {@link #MAX_RECONNECT_GAP_IN_SECONDS} 30s
   * @return this, for chaining
   */
  public PusherBuilder setMaxReconnectGapInSeconds(int maxReconnectGapInSeconds) {
    this.maxReconnectGapInSeconds = maxReconnectGapInSeconds;
    return this;
  }

  public long getPongTimeout() {
    return pongTimeout;
  }

  /**
   * Construct the URL for the WebSocket connection based on the options
   * previous set on this object and the provided API key
   *
   * @param apiKey The API key
   * @return the WebSocket URL
   */
  public static URI createURI(final boolean encrypted, final String apiKey) {
    return createURI(encrypted, null, apiKey);
  }

  /**
   * Construct the URL for the WebSocket connection based on the options
   * previous set on this object and the provided API key
   *
   * @param apiKey The API key
   * @return the WebSocket URL
   */
  public static URI createURI(final boolean encrypted, final String cluster, final String apiKey) {
    return URI.create((encrypted ? WSS_SCHEME : WS_SCHEME) + "://"
        + (cluster == null ? "ws" : "ws-" + cluster)
        + "." + PUSHER_DOMAIN
        + ":" + (encrypted ? WSS_PORT : WS_PORT)
        + "/app/" + apiKey
        + URI_PARAMS);
  }

  /**
   * @return the maximum reconnection gap in seconds
   */
  public int getMaxReconnectGapInSeconds() {
    return maxReconnectGapInSeconds;
  }

  private static String readVersionFromProperties() {
    try (final var inStream = PusherBuilder.class.getResourceAsStream("/pusher.properties")) {
      final var p = new Properties();
      p.load(inStream);
      final var version = (String) p.get("version");
      // If the properties file contents indicates the version is being run
      // from source then replace with a dev indicator. Otherwise the Pusher
      // Socket API will reject the connection.
      if (version != null && !version.isEmpty()) {
        return version.equals(SRC_LIB_DEV_VERSION) ? LIB_DEV_VERSION : version;
      }
    } catch (final Exception e) {
      // Fall back to fixed value
    }
    return "0.0.0";
  }
}
