package com.pusher.client;

import com.pusher.client.channel.*;
import com.pusher.client.channel.impl.ChannelImpl;
import com.pusher.client.channel.impl.ChannelManager;
import com.pusher.client.channel.impl.PresenceChannelImpl;
import com.pusher.client.channel.impl.PrivateChannelImpl;
import com.pusher.client.connection.ConnectionEventListener;
import com.pusher.client.connection.ConnectionManager;
import com.pusher.client.connection.ConnectionState;
import com.pusher.client.connection.impl.InternalConnectionManager;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.concurrent.Executor;

import static com.pusher.client.connection.ConnectionState.ALL;

/**
 * This class is the main entry point for accessing Pusher.
 *
 * <p>
 * By creating a new {@link Pusher} instance and calling {@link
 * Pusher#connect()} a connectionManager to Pusher is established.
 * </p>
 *
 * <p>
 * Subscriptions for data are represented by
 * {@link com.pusher.client.channel.Channel} objects, or subclasses thereof.
 * Subscriptions are created by calling {@link Pusher#subscribe(String)},
 * {@link Pusher#subscribePrivate(String)},
 * {@link Pusher#subscribePresence(String)} or one of the overloads.
 * </p>
 */
public final class Pusher implements Client {

  private final Executor listenerExecutor;
  private final InternalConnectionManager connectionManager;
  private final ChannelManager channelManager;
  private final Authorizer authorizer;

  Pusher(final Executor listenerExecutor,
         final ChannelManager channelManager,
         final InternalConnectionManager connectionManager,
         final Authorizer authorizer) {
    this.listenerExecutor = listenerExecutor;
    this.channelManager = channelManager;
    this.connectionManager = connectionManager;
    this.authorizer = authorizer;
  }

  public static PusherBuilder build() {
    return new PusherBuilder();
  }

  /* ConnectionManager methods */

  /**
   * Gets the underlying {@link ConnectionManager} object that is being used by this
   * instance of {@linkplain Pusher}.
   *
   * @return The {@link ConnectionManager} object.
   */
  public ConnectionManager getConnectionManager() {
    return connectionManager;
  }

  /**
   * Connects to Pusher. Any {@link ConnectionEventListener}s that have
   * already been registered using the
   * {@link ConnectionManager#bind(ConnectionState, ConnectionEventListener)} method
   * will receive connectionManager events.
   *
   * <p>Calls are ignored (a connectionManager is not attempted) if the {@link ConnectionManager#getState()} is not {@link com.pusher.client.connection.ConnectionState#DISCONNECTED}.</p>
   */
  @Override
  public void connect() {
    connect(null);
  }

  /**
   * Connects to Pusher. Any {@link ConnectionEventListener}s that have
   * already been registered using the
   * {@link ConnectionManager#bind(ConnectionState, ConnectionEventListener)} method
   * will receive connectionManager events.
   *
   * <p>Calls are ignored (a connectionManager is not attempted) if the {@link ConnectionManager#getState()} is not {@link com.pusher.client.connection.ConnectionState#DISCONNECTED}.</p>
   */
  @Override
  public void connect(final ConnectionEventListener eventListener) {
    connect(eventListener, EnumSet.noneOf(ConnectionState.class));
  }

  /**
   * Binds a {@link ConnectionEventListener} to the specified events and then
   * connects to Pusher. This is equivalent to binding a
   * {@link ConnectionEventListener} using the
   * {@link ConnectionManager#bind(ConnectionState, ConnectionEventListener)} method
   * before connecting.
   *
   * <p>Calls are ignored (a connectionManager is not attempted) if the {@link ConnectionManager#getState()} is not {@link com.pusher.client.connection.ConnectionState#DISCONNECTED}.</p>
   *
   * @param eventListener    A {@link ConnectionEventListener} that will receive connectionManager
   *                         events. This can be null if you are not interested in
   *                         receiving connectionManager events, in which case you should call
   *                         {@link #connect()} instead of this method.
   * @param connectionStates An optional list of {@link ConnectionState}s to bind your
   *                         {@link ConnectionEventListener} to before connecting to
   *                         Pusher. If you do not specify any {@link ConnectionState}s
   *                         then your {@link ConnectionEventListener} will be bound to all
   *                         connectionManager events.
   *                         with {@link ConnectionState#ALL}.
   * @throws IllegalArgumentException If the {@link ConnectionEventListener} is null and at least
   *                                  one connectionManager state has been specified.
   */
  @Override
  public void connect(final ConnectionEventListener eventListener, final Collection<ConnectionState> connectionStates) {
    if (eventListener != null) {
      if (connectionStates.isEmpty()) {
        connectionManager.bind(ALL, eventListener);
      } else {
        for (final var state : connectionStates) {
          connectionManager.bind(state, eventListener);
        }
      }
    } else if (!connectionStates.isEmpty()) {
      throw new IllegalArgumentException(
          "Cannot bind to connectionManager states with a null connectionManager event listener");
    }
    connectionManager.connect();
  }

  /**
   * Disconnect from Pusher.
   *
   * <p>
   * Calls are ignored if the {@link ConnectionManager#getState()}, retrieved from {@link Pusher#getConnectionManager}, is not
   * {@link com.pusher.client.connection.ConnectionState#CONNECTED}.
   * </p>
   */
  public void disconnect() {
    if (connectionManager.getState() == ConnectionState.CONNECTED) {
      connectionManager.disconnect();
    }
  }

  /* Subscription methods */

  /**
   * Subscribes to a public {@link Channel}.
   * <p>
   * Note that subscriptions should be registered only once with a Pusher
   * instance. Subscriptions are persisted over disconnection and
   * re-registered with the server automatically on reconnection. This means
   * that subscriptions may also be registered before connect() is called,
   * they will be initiated on connectionManager.
   *
   * @param channelName The channelName of the {@link Channel} to subscribe to.
   * @return The {@link Channel} object representing your subscription.
   */
  @Override
  public Channel subscribe(final String channelName) {
    return subscribe(channelName, null);
  }

  /**
   * Binds a {@link ChannelEventListener} to the specified events and then
   * subscribes to a public {@link Channel}.
   *
   * @param channelName          The channelName of the {@link Channel} to subscribe to.
   * @param channelEventListener A {@link ChannelEventListener} to be informed of Pusher channel protocol events and subscription data events if no event listener is bound.
   * @return The {@link Channel} object representing your subscription.
   * @throws IllegalArgumentException If any of the following are true:
   *                                  <ul>
   *                                  <li>The channel channelName is null.</li>
   *                                  <li>You are already subscribed to this channel.</li>
   *                                  <li>The channel channelName starts with "private-". If you want to
   *                                  subscribe to a private channel, call
   *                                  {@link #subscribePrivate(String, PrivateChannelEventListener, DataEventListener, Iterable)}
   *                                  instead of this method.</li>
   *                                  <li>At least one of the specified event names is null.</li>
   *                                  <li>You have specified at least one event channelName and your
   *                                  {@link ChannelEventListener} is null.</li>
   *                                  </ul>
   */
  @Override
  public Channel subscribe(final String channelName, final ChannelEventListener channelEventListener) {
    final var channel = new ChannelImpl(channelName, listenerExecutor);
    channelManager.subscribeTo(channel, channelEventListener, null, Collections.emptyList());
    return channel;
  }

  /**
   * Binds a {@link ChannelEventListener} to the specified events and then
   * subscribes to a public {@link Channel}.
   *
   * @param channelName          The channelName of the {@link Channel} to subscribe to.
   * @param channelEventListener A {@link ChannelEventListener} to be informed of Pusher channel protocol events and subscription data events if no event listener is bound.
   * @param dataEventListener    A {@link DataEventListener} to receive events. This can be
   *                             null if you don't want to bind a listener at subscription
   *                             time, in which case you should call {@link #subscribe(String, ChannelEventListener)}
   *                             instead of this method.
   * @param eventNames           An optional list of event names to bind your
   *                             {@link DataEventListener} to before subscribing.
   * @return The {@link Channel} object representing your subscription.
   * @throws IllegalArgumentException If any of the following are true:
   *                                  <ul>
   *                                  <li>The channel channelName is null.</li>
   *                                  <li>You are already subscribed to this channel.</li>
   *                                  <li>The channel channelName starts with "private-". If you want to
   *                                  subscribe to a private channel, call
   *                                  {@link #subscribePrivate(String, PrivateChannelEventListener, DataEventListener, Iterable)}
   *                                  instead of this method.</li>
   *                                  <li>At least one of the specified event names is null.</li>
   *                                  <li>You have specified at least one event channelName and your
   *                                  {@link ChannelEventListener} is null.</li>
   *                                  </ul>
   */
  @Override
  public Channel subscribe(final String channelName,
                           final ChannelEventListener channelEventListener,
                           final DataEventListener dataEventListener,
                           final Iterable<String> eventNames) {
    final var channel = new ChannelImpl(channelName, listenerExecutor);
    channelManager.subscribeTo(channel, channelEventListener, dataEventListener, eventNames);
    return channel;
  }

  /**
   * Subscribes to a {@link com.pusher.client.channel.PrivateChannel} which
   * requires authentication.
   *
   * @param channelName The channelName of the channel to subscribe to.
   * @return A new {@link com.pusher.client.channel.PrivateChannel}
   * representing the subscription.
   * @throws IllegalStateException if a {@link com.pusher.client.Authorizer} has not been set.
   */
  @Override
  public PrivateChannel subscribePrivate(final String channelName) {
    return subscribePrivate(channelName, null);
  }

  /**
   * Subscribes to a {@link com.pusher.client.channel.PrivateChannel} which
   * requires authentication.
   *
   * @param channelName          The channelName of the channel to subscribe to.
   * @param channelEventListener A {@link ChannelEventListener} to be informed of Pusher channel protocol events and subscription data events if no event listener is bound.
   * @return A new {@link com.pusher.client.channel.PrivateChannel} representing the subscription.
   * @throws IllegalStateException if a {@link com.pusher.client.Authorizer} has not been set.
   */
  @Override
  public PrivateChannel subscribePrivate(final String channelName,
                                         final PrivateChannelEventListener channelEventListener) {
    return subscribePrivate(channelName, channelEventListener, null, Collections.emptyList());
  }

  /**
   * Subscribes to a {@link com.pusher.client.channel.PrivateChannel} which
   * requires authentication.
   *
   * @param channelName          The channelName of the channel to subscribe to.
   * @param channelEventListener A listener to be informed of Pusher channel protocol events and subscription data events if no event listener is bound.
   * @param dataEventListener    A listener to be informed of subscription data events.
   * @param eventNames           An optional list of names of events to be bound to on the channel. The equivalent of calling {@link com.pusher.client.channel.Channel#bind(String, DataEventListener)} one or more times.
   * @return A new {@link com.pusher.client.channel.PrivateChannel} representing the subscription.
   * @throws IllegalStateException if a {@link com.pusher.client.Authorizer} has not been set.
   */
  @Override
  public PrivateChannel subscribePrivate(final String channelName,
                                         final PrivateChannelEventListener channelEventListener,
                                         final DataEventListener dataEventListener,
                                         final Iterable<String> eventNames) {
    if (authorizer == null) {
      throw new IllegalStateException(
          "Cannot subscribe to a private or presence channel because no Authorizer has been set. Call PusherBuilder.setAuthorizer() before connecting to Pusher");
    }
    final var channel = new PrivateChannelImpl(connectionManager, channelName, authorizer, listenerExecutor);
    channelManager.subscribeTo(channel, channelEventListener, dataEventListener, eventNames);
    return channel;
  }

  /**
   * Subscribes to a {@link com.pusher.client.channel.PresenceChannel} which
   * requires authentication.
   *
   * @param channelName The channelName of the channel to subscribe to.
   * @return A new {@link com.pusher.client.channel.PresenceChannel}
   * representing the subscription.
   * @throws IllegalStateException if a {@link com.pusher.client.Authorizer} has not been set.
   */
  @Override
  public PresenceChannel subscribePresence(final String channelName) {
    return subscribePresence(channelName, null);
  }

  /**
   * Subscribes to a {@link com.pusher.client.channel.PresenceChannel} which
   * requires authentication.
   *
   * @param channelName          The channelName of the channel to subscribe to.
   * @param channelEventListener A listener to be informed of Pusher channel protocol, including presence-specific events, and subscription data events if no event listener is bound.
   * @return A new {@link com.pusher.client.channel.PresenceChannel} representing the subscription.
   * @throws IllegalStateException if a {@link com.pusher.client.Authorizer} has not been set.
   */
  @Override
  public PresenceChannel subscribePresence(final String channelName,
                                           final PresenceChannelEventListener channelEventListener) {
    return subscribePresence(channelName, channelEventListener, null, Collections.emptyList());
  }

  /**
   * Subscribes to a {@link com.pusher.client.channel.PresenceChannel} which
   * requires authentication.
   *
   * @param channelName          The channelName of the channel to subscribe to.
   * @param channelEventListener A listener to be informed of Pusher channel protocol, including presence-specific events, and subscription data events if no event listener is bound.
   * @param dataEventListener    A listener to be informed of subscription data events.
   * @param eventNames           An optional list of names of events to be bound to on the channel. The equivalent of calling {@link com.pusher.client.channel.Channel#bind(String, DataEventListener)} one or more times.
   * @return A new {@link com.pusher.client.channel.PresenceChannel} representing the subscription.
   * @throws IllegalStateException if a {@link com.pusher.client.Authorizer} has not been set.
   */
  @Override
  public PresenceChannel subscribePresence(final String channelName,
                                           final PresenceChannelEventListener channelEventListener,
                                           final DataEventListener dataEventListener,
                                           final Iterable<String> eventNames) {
    if (authorizer == null) {
      throw new IllegalStateException(
          "Cannot subscribe to a private or presence channel because no Authorizer has been set. Call PusherBuilder.setAuthorizer() before connecting to Pusher");
    }
    final var channel = new PresenceChannelImpl(connectionManager, channelName, authorizer, listenerExecutor);
    channelManager.subscribeTo(channel, channelEventListener, dataEventListener, eventNames);
    return channel;
  }

  /**
   * Unsubscribes from a channel using via the channelName of the channel.
   *
   * @param channelName the channelName of the channel to be unsubscribed from.
   */
  public void unsubscribe(final String channelName) {
    channelManager.unsubscribeFrom(channelName);
  }

  /* implementation detail */

  /**
   * @param channelName The channelName of the public channel to be retrieved
   * @return A public channel, or null if it could not be found
   * @throws IllegalArgumentException if you try to retrieve a private or presence channel.
   */
  public Channel getChannel(final String channelName) {
    return channelManager.getChannel(channelName);
  }

  /**
   * @param channelName The channelName of the private channel to be retrieved
   * @return A private channel, or null if it could not be found
   * @throws IllegalArgumentException if you try to retrieve a public or presence channel.
   */
  public PrivateChannel getPrivateChannel(final String channelName) {
    return channelManager.getPrivateChannel(channelName);
  }

  /**
   * @param channelName The channelName of the presence channel to be retrieved
   * @return A presence channel, or null if it could not be found
   * @throws IllegalArgumentException if you try to retrieve a public or private channel.
   */
  public PresenceChannel getPresenceChannel(final String channelName) {
    return channelManager.getPresenceChannel(channelName);
  }
}
