package com.pusher.client.util;

import com.pusher.client.AuthorizationFailureException;
import com.pusher.client.Authorizer;
import jdk.incubator.http.HttpClient;
import jdk.incubator.http.HttpRequest;

import java.io.IOException;
import java.net.URI;
import java.util.Collections;
import java.util.Map;

import static jdk.incubator.http.HttpRequest.BodyPublisher.fromString;
import static jdk.incubator.http.HttpResponse.BodyHandler.asString;

/**
 * Used to authenticate a {@link com.pusher.client.channel.PrivateChannel
 * private} or {@link com.pusher.client.channel.PresenceChannel presence}
 * channel subscription.
 *
 * <p>
 * Makes an HTTP request to a defined HTTP endpoint. Expects an authentication
 * token to be returned.
 * </p>
 * <p>
 * <p>
 * For more information see the <a
 * href="http://pusher.com/docs/authenticating_users">Authenticating Users
 * documentation</a>.
 */
public final class HttpAuthorizer implements Authorizer {

  private final HttpClient httpClient;
  private final URI endPoint;
  private Map<String, String> mHeaders;
  private ConnectionFactory mConnectionFactory;

  public HttpAuthorizer(final String endPoint) {
    this(null, endPoint, new UrlEncodedConnectionFactory());
    this.mHeaders = Collections.emptyMap();
  }

  /**
   * Creates a new authorizer.
   *
   * @param endPoint The endpoint to be called when authenticating.
   */
  public HttpAuthorizer(final HttpClient httpClient, final String endPoint) {
    this(httpClient, endPoint, new UrlEncodedConnectionFactory());
  }

  /**
   * Creates a new authorizer.
   *
   * @param endPoint          The endpoint to be called when authenticating.
   * @param connectionFactory a custom connection factory to be used for building the connection
   */
  public HttpAuthorizer(final HttpClient httpClient, final String endPoint, final ConnectionFactory connectionFactory) {
    this.httpClient = httpClient;
    this.endPoint = URI.create(endPoint);
    this.mConnectionFactory = connectionFactory;
  }

  /**
   * Set additional headers to be sent as part of the request.
   *
   * @param headers A map of headers
   */
  public void setHeaders(final Map<String, String> headers) {
    mHeaders = headers == null ? Collections.emptyMap() : headers;
  }

  /**
   * Identifies if the HTTP request will be sent over HTTPS.
   *
   * @return true if the endpoint protocol is 'https'
   */
  Boolean isSSL() {
    return endPoint.getScheme().equals("https");
  }

  @Override
  public String authorize(final String channelName, final String socketId) throws AuthorizationFailureException {
    mConnectionFactory.setChannelName(channelName);
    mConnectionFactory.setSocketId(socketId);
    final var postRequest = HttpRequest.newBuilder(endPoint);
    mHeaders.forEach(postRequest::header);
    postRequest.headers(
        "Content-Type", mConnectionFactory.getContentType(),
        "charset", mConnectionFactory.getCharset());
    try {
      final var response = (httpClient == null ? HttpClient.newHttpClient() : httpClient)
          .send(postRequest.POST(fromString(mConnectionFactory.getBody())).build(), asString());
      final var responseHttpStatus = response.statusCode();
      if (responseHttpStatus == 200 || responseHttpStatus == 201) {
        return response.body();
      }
      throw new AuthorizationFailureException(response.toString());
    } catch (final InterruptedException e) {
      Thread.interrupted();
      throw new RuntimeException(e);
    } catch (final IOException e) {
      throw new AuthorizationFailureException(e);
    }
  }
}
