package com.pusher.client.channel.impl;

import com.pusher.client.channel.ChannelEventListener;
import com.pusher.client.channel.ChannelState;
import com.pusher.client.channel.SubscriptionEventListener;
import com.pusher.client.util.Factory;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

public class ChannelImpl implements InternalChannel {

  private static final String INTERNAL_EVENT_PREFIX = "pusher_internal:";
  static final String SUBSCRIPTION_SUCCESS_EVENT = "pusher_internal:subscription_succeeded";

  protected final String channelName;
  private final Map<String, Set<SubscriptionEventListener>> eventNameToListenerMap;
  protected volatile ChannelState state;
  private ChannelEventListener eventListener;
  private final Factory factory;

  public ChannelImpl(final String channelName, final Factory factory) {
    if (channelName == null) {
      throw new IllegalArgumentException("Cannot subscribe to a channel with a null channelName");
    }
    for (final var disallowedPattern : getDisallowedNameExpressions()) {
      if (channelName.matches(disallowedPattern)) {
        throw new IllegalArgumentException(
            "Channel channelName " + channelName
                + " is invalid. Private channel names must start with \"private-\" and presence channel names must start with \"presence-\"");
      }
    }
    this.channelName = channelName;
    this.eventNameToListenerMap = new ConcurrentHashMap<>();
    this.state = ChannelState.INITIAL;
    this.factory = factory;
  }

  /* Channel implementation */

  @Override
  public String getChannelName() {
    return channelName;
  }

  @Override
  public void bind(final String eventName, final SubscriptionEventListener listener) {
    validateArguments(eventName, listener);
    eventNameToListenerMap.computeIfAbsent(eventName, _eventName -> Collections.newSetFromMap(new ConcurrentHashMap<>())).add(listener);
  }

  @Override
  public void unbind(final String eventName, final SubscriptionEventListener listener) {
    validateArguments(eventName, listener);
    final var listeners = eventNameToListenerMap.get(eventName);
    if (listeners != null) {
      listeners.remove(listener);
    }
  }

  @Override
  public boolean isSubscribed() {
    return state == ChannelState.SUBSCRIBED;
  }

  /* InternalChannel implementation */

  @Override
  public void onMessage(final String event, final String rawJson) {
    if (event.equals(SUBSCRIPTION_SUCCESS_EVENT)) {
      updateState(ChannelState.SUBSCRIBED);
      return;
    }
    final var data = getData(rawJson);
    final var listeners = eventNameToListenerMap.get(event);
    if (listeners == null) {
      return;
    }
    for (final var listener : listeners) {
      factory.queueOnEventThread(() -> listener.onEvent(channelName, event, data));
    }
  }

  private static String getData(final String json) {
    var index = json.indexOf("\"data\":");
    if (index < 0) {
      return null;
    }
    int begin;
    for (index += 7; ; ) {
      final var ch = json.charAt(index++);
      if (ch == '"') {
        begin = index;
        break;
      }
    }
    for (char prevCh = json.charAt(index), ch = json.charAt(++index); ; ) {
      if (ch == '"' && prevCh != '\\') {
        return json.substring(begin, index).replace("\\\"", "\"");
      }
      prevCh = ch;
      ch = json.charAt(++index);
    }
  }

  @Override
  public String toSubscribeMessage() {
    return "{\"event\":\"pusher:subscribe\",\"data\":{\"channel\":\"" + channelName + "\"}}";
  }

  @Override
  public String toUnsubscribeMessage() {
    return "{\"event\":\"pusher:unsubscribe\",\"data\":{\"channel\":\"" + channelName + "\"}}";
  }

  @Override
  public void updateState(final ChannelState state) {
    this.state = state;
    if (state == ChannelState.SUBSCRIBED && eventListener != null) {
      factory.queueOnEventThread(() -> eventListener.onSubscriptionSucceeded(channelName));
    }
  }

  /* Comparable implementation */

  @Override
  public void setEventListener(final ChannelEventListener listener) {
    eventListener = listener;
  }

  @Override
  public ChannelEventListener getEventListener() {
    return eventListener;
  }

  @Override
  public int compareTo(final InternalChannel other) {
    return channelName.compareTo(other.getChannelName());
  }

  /* implementation detail */

  @Override
  public String toString() {
    return String.format("[Public Channel: channelName=%s]", channelName);
  }

  protected String[] getDisallowedNameExpressions() {
    return new String[]{"^private-.*", "^presence-.*"};
  }

  private void validateArguments(final String eventName, final SubscriptionEventListener listener) {
    if (eventName == null) {
      throw new IllegalArgumentException("Cannot bind or unbind to channel " + channelName + " with a null event channelName");
    }
    if (listener == null) {
      throw new IllegalArgumentException("Cannot bind or unbind to channel " + channelName + " with a null listener");
    }
    if (eventName.startsWith(INTERNAL_EVENT_PREFIX)) {
      throw new IllegalArgumentException("Cannot bind or unbind channel " + channelName
          + " with an internal event channelName such as " + eventName);
    }
    if (state == ChannelState.UNSUBSCRIBED) {
      throw new IllegalStateException(
          "Cannot bind or unbind to events on a channel that has been unsubscribed. Call Pusher.subscribe() to resubscribe to this channel");
    }
  }
}
