/*
 * Decompiled with CFR 0.152.
 */
package com.pusher.client.connection.websocket;

import com.pusher.client.connection.ConnectionEventListener;
import com.pusher.client.connection.ConnectionState;
import com.pusher.client.connection.ConnectionStateChange;
import com.pusher.client.connection.impl.InternalConnection;
import com.pusher.client.util.Factory;
import com.pusher.client.util.PusherJsonParser;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import jdk.incubator.http.HttpClient;
import jdk.incubator.http.WebSocket;

public final class WebSocketConnection
implements InternalConnection,
WebSocket.Listener {
    private static final System.Logger log = System.getLogger(WebSocketConnection.class.getName());
    private static final String INTERNAL_EVENT_PREFIX = "pusher:";
    private static final String PING_EVENT_SERIALIZED = "{\"event\": \"pusher:ping\"}";
    private final HttpClient httpClient;
    private final Factory factory;
    private final ActivityTimer activityTimer;
    private final Map<ConnectionState, Set<ConnectionEventListener>> eventListeners = new ConcurrentHashMap<ConnectionState, Set<ConnectionEventListener>>();
    private final URI webSocketUri;
    private final int maxReconnectionAttempts;
    private final int maxReconnectionGap;
    private volatile ConnectionState state = ConnectionState.DISCONNECTED;
    private WebSocket underlyingConnection;
    private String socketId;
    private int reconnectAttempts = 0;
    private StringBuilder partialBuilder;

    public WebSocketConnection(HttpClient httpClient, String url, long activityTimeout, long pongTimeout, int maxReconnectionAttempts, int maxReconnectionGap, Factory factory) throws URISyntaxException {
        this.httpClient = httpClient;
        this.webSocketUri = new URI(url);
        this.activityTimer = new ActivityTimer(activityTimeout, pongTimeout);
        this.maxReconnectionAttempts = maxReconnectionAttempts;
        this.maxReconnectionGap = maxReconnectionGap;
        this.factory = factory;
        for (ConnectionState state : ConnectionState.values()) {
            this.eventListeners.put(state, Collections.newSetFromMap(new ConcurrentHashMap()));
        }
    }

    @Override
    public void connect() {
        this.factory.queueOnEventThread(() -> {
            if (this.state == ConnectionState.DISCONNECTED) {
                this.tryConnecting();
            }
        });
    }

    private void tryConnecting() {
        try {
            CompletableFuture futureWebSocket = (this.httpClient == null ? HttpClient.newHttpClient() : this.httpClient).newWebSocketBuilder().buildAsync(this.webSocketUri, (WebSocket.Listener)this);
            this.updateState(ConnectionState.CONNECTING);
            this.underlyingConnection = (WebSocket)futureWebSocket.join();
        }
        catch (RuntimeException e) {
            this.sendErrorToAllListeners("Error connecting to WebSocket server.", null, e);
        }
    }

    @Override
    public void disconnect() {
        this.factory.queueOnEventThread(() -> {
            if (this.state == ConnectionState.CONNECTED) {
                this.updateState(ConnectionState.DISCONNECTING);
                this.underlyingConnection.sendClose(1000, "");
            }
        });
    }

    @Override
    public void bind(ConnectionState state, ConnectionEventListener eventListener) {
        this.eventListeners.get((Object)state).add(eventListener);
    }

    @Override
    public boolean unbind(ConnectionState state, ConnectionEventListener eventListener) {
        return this.eventListeners.get((Object)state).remove(eventListener);
    }

    @Override
    public ConnectionState getState() {
        return this.state;
    }

    @Override
    public void sendMessage(String message) {
        this.factory.queueOnEventThread(() -> {
            try {
                if (this.state == ConnectionState.CONNECTED) {
                    this.underlyingConnection.sendText((CharSequence)message, true);
                } else {
                    this.sendErrorToAllListeners("Cannot send a message while in " + this.state + " state", null, null);
                }
            }
            catch (Exception e) {
                this.sendErrorToAllListeners("An exception occurred while sending message [" + message + "]", null, e);
            }
        });
    }

    @Override
    public String getSocketId() {
        return this.socketId;
    }

    private void updateState(ConnectionState newState) {
        log.log(System.Logger.Level.TRACE, "State transition requested, current [" + this.state + "], new [" + newState + "]");
        ConnectionStateChange change = new ConnectionStateChange(this.state, newState);
        this.state = newState;
        HashSet dedupe = new HashSet();
        this.eventListeners.get((Object)ConnectionState.ALL).stream().filter(dedupe::add).forEach(listener -> listener.onConnectionStateChange(change));
        this.eventListeners.get((Object)newState).stream().filter(dedupe::add).forEach(listener -> listener.onConnectionStateChange(change));
    }

    private void handleEvent(String rawJson) {
        String event = PusherJsonParser.getJsonValue(rawJson, "\"event\":");
        if (event == null) {
            throw new IllegalArgumentException("Message does not contain an event field: " + rawJson);
        }
        if (event.startsWith(INTERNAL_EVENT_PREFIX)) {
            this.handleInternalEvent(event, rawJson);
            return;
        }
        this.factory.getChannelManager().onMessage(event, rawJson);
    }

    private void handleInternalEvent(String event, String rawJson) {
        if (event.equals("pusher:connection_established")) {
            this.handleConnectionMessage(rawJson);
        } else if (event.equals("pusher:error")) {
            this.handleError(rawJson);
        }
    }

    private void handleConnectionMessage(String rawJson) {
        this.socketId = PusherJsonParser.getEscapedJsonValue(rawJson, "\\\"socket_id\\\":");
        if (this.state != ConnectionState.CONNECTED) {
            this.updateState(ConnectionState.CONNECTED);
        }
        this.reconnectAttempts = 0;
    }

    private void handleError(String rawJson) {
        String message = PusherJsonParser.getJsonValue(rawJson, "\"message\":");
        String code = PusherJsonParser.getJsonValue(rawJson, "\"code\":");
        this.sendErrorToAllListeners(message, code, null);
    }

    private void sendErrorToAllListeners(String message, String code, Throwable e) {
        HashSet dedupe = new HashSet();
        this.eventListeners.values().stream().flatMap(Collection::stream).filter(dedupe::add).forEach(listener -> this.factory.queueOnEventThread(() -> listener.onError(message, code, e)));
    }

    public void onOpen(WebSocket webSocket) {
        webSocket.request(Long.MAX_VALUE);
    }

    public CompletionStage<?> onText(WebSocket webSocket, CharSequence message, WebSocket.MessagePart part) {
        this.activityTimer.activity();
        switch (part) {
            case WHOLE: {
                String msgCopy = message.toString();
                this.factory.queueOnEventThread(() -> this.handleEvent(msgCopy));
                return null;
            }
            case FIRST: {
                if (this.partialBuilder == null) {
                    this.partialBuilder = new StringBuilder(message);
                    return null;
                }
                this.partialBuilder.setLength(0);
                this.partialBuilder.append(message);
                return null;
            }
            case PART: {
                this.partialBuilder.append(message);
                return null;
            }
            case LAST: {
                String fullMessage = this.partialBuilder.append(message).toString();
                this.factory.queueOnEventThread(() -> this.handleEvent(fullMessage));
                return null;
            }
        }
        throw new IllegalStateException(part + " WebSocket messages are not currently supported. Please open an issue with replication details.");
    }

    public CompletionStage<?> onClose(WebSocket webSocket, int statusCode, String reason) {
        if (this.state == ConnectionState.DISCONNECTED || this.state == ConnectionState.RECONNECTING) {
            log.log(System.Logger.Level.WARNING, "Received close from underlying socket when already disconnected.  Close code [" + statusCode + "], Reason [" + reason + "]");
            this.partialBuilder = null;
            return null;
        }
        if (this.state == ConnectionState.CONNECTED || this.state == ConnectionState.CONNECTING) {
            if (this.reconnectAttempts < this.maxReconnectionAttempts) {
                this.tryReconnecting();
            } else {
                this.updateState(ConnectionState.DISCONNECTING);
                this.cancelTimeoutsAndTransitonToDisconnected();
            }
        } else if (this.state == ConnectionState.DISCONNECTING) {
            this.cancelTimeoutsAndTransitonToDisconnected();
        }
        this.partialBuilder = null;
        return null;
    }

    private void tryReconnecting() {
        ++this.reconnectAttempts;
        this.updateState(ConnectionState.RECONNECTING);
        long reconnectInterval = Math.min(this.maxReconnectionGap, this.reconnectAttempts * this.reconnectAttempts);
        this.factory.getTimers().schedule(this::tryConnecting, reconnectInterval, TimeUnit.SECONDS);
    }

    private void cancelTimeoutsAndTransitonToDisconnected() {
        this.activityTimer.cancelTimeouts();
        this.factory.queueOnEventThread(() -> {
            this.updateState(ConnectionState.DISCONNECTED);
            this.factory.shutdownThreads();
        });
    }

    public void onError(WebSocket webSocket, Throwable error) {
        this.factory.queueOnEventThread(() -> this.sendErrorToAllListeners("An exception was thrown by the WebSocket", null, error));
    }

    private class ActivityTimer {
        private final long activityTimeout;
        private final long pongTimeout;
        private Future<?> pingTimer;
        private Future<?> pongTimer;

        ActivityTimer(long activityTimeout, long pongTimeout) {
            this.activityTimeout = activityTimeout;
            this.pongTimeout = pongTimeout;
        }

        synchronized void activity() {
            if (this.pongTimer != null) {
                this.pongTimer.cancel(true);
            }
            if (this.pingTimer != null) {
                this.pingTimer.cancel(false);
            }
            this.pingTimer = WebSocketConnection.this.factory.getTimers().schedule(() -> {
                log.log(System.Logger.Level.TRACE, "Sending ping");
                WebSocketConnection.this.sendMessage(WebSocketConnection.PING_EVENT_SERIALIZED);
                this.schedulePongCheck();
            }, this.activityTimeout, TimeUnit.MILLISECONDS);
        }

        synchronized void cancelTimeouts() {
            if (this.pingTimer != null) {
                this.pingTimer.cancel(false);
            }
            if (this.pongTimer != null) {
                this.pongTimer.cancel(false);
            }
        }

        private synchronized void schedulePongCheck() {
            if (this.pongTimer != null) {
                this.pongTimer.cancel(false);
            }
            this.pongTimer = WebSocketConnection.this.factory.getTimers().schedule(() -> {
                log.log(System.Logger.Level.TRACE, "Timed out awaiting pong from server - disconnecting");
                WebSocketConnection.this.disconnect();
                WebSocketConnection.this.onClose(WebSocketConnection.this.underlyingConnection, -1, "Pong timeout");
            }, this.pongTimeout, TimeUnit.MILLISECONDS);
        }
    }
}

