package com.pusher.client.channel.impl;

import com.pusher.client.AuthorizationFailureException;
import com.pusher.client.Authorizer;
import com.pusher.client.channel.PresenceChannel;
import com.pusher.client.channel.PresenceChannelEventListener;
import com.pusher.client.channel.SubscriptionEventListener;
import com.pusher.client.channel.User;
import com.pusher.client.connection.impl.InternalConnection;
import com.pusher.client.util.Factory;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import static com.pusher.client.util.PusherJsonParser.getEscapedJsonValue;

public class PresenceChannelImpl extends PrivateChannelImpl implements PresenceChannel {

  private static final String MEMBER_ADDED_EVENT = "pusher_internal:member_added";
  private static final String MEMBER_REMOVED_EVENT = "pusher_internal:member_removed";

  private final Map<String, User> idToUserMap = new ConcurrentHashMap<>();
  private String myUserID;

  public PresenceChannelImpl(final InternalConnection connection,

                             final String channelName,
                             final Authorizer authorizer, final Factory factory) {
    super(connection, channelName, authorizer, factory);
  }

  /* PresenceChannel implementation */

  @Override
  public Set<User> getUsers() {
    return Set.copyOf(idToUserMap.values());
  }

  @Override
  public User getMe() {
    return idToUserMap.get(myUserID);
  }

  /* Base class overrides */

  @Override
  public void onMessage(final String event, final String rawJson) {
    super.onMessage(event, rawJson);
    switch (event) {
      case SUBSCRIPTION_SUCCESS_EVENT:
        handleSubscriptionSuccessfulMessage(rawJson);
        break;
      case MEMBER_ADDED_EVENT:
        handleMemberAddedEvent(rawJson);
        break;
      case MEMBER_REMOVED_EVENT:
        handleMemberRemovedEvent(rawJson);
        break;
    }
  }

  @Override
  @SuppressWarnings("rawtypes")
  public String toSubscribeMessage() {
    final var authResponse = getAuthResponse();
    try {
      myUserID = getEscapedJsonValue(authResponse, "\\\"user_id\\\"");
      return "{\"event\":\"pusher:subscribe\",\"data\":{\"channel\":\"" + channelName + "\"," + authResponse.substring(1) + "}";
    } catch (final Exception e) {
      throw new AuthorizationFailureException("Unable to parse response from Authorizer: " + authResponse, e);
    }
  }

  @Override
  public void bind(final String eventName, final SubscriptionEventListener listener) {
    if (!(listener instanceof PresenceChannelEventListener)) {
      throw new IllegalArgumentException(
          "Only instances of PresenceChannelEventListener can be bound to a presence channel");
    }
    super.bind(eventName, listener);
  }

  @Override
  protected String[] getDisallowedNameExpressions() {
    return new String[]{"^(?!presence-).*"};
  }

  @Override
  public String toString() {
    return String.format("[Presence Channel: channelName=%s]", channelName);
  }

  @SuppressWarnings({"rawtypes", "unchecked"})
  private void handleSubscriptionSuccessfulMessage(final String rawJson) {
    parseUsers(rawJson);
    final var listener = getEventListener();
    if (listener != null) {
      final var presenceListener = (PresenceChannelEventListener) listener;
      presenceListener.onUsersInformationReceived(getChannelName(), getUsers());
    }
  }

  private void parseUsers(final String json) {
    final var hashIndex = json.indexOf("\\\"hash\\\"");
    if (hashIndex < 0) {
      return;
    }
    final var open = json.indexOf("{", hashIndex + 9);
    for (var index = open; ; ) {
      final var beginKey = json.indexOf("\\\"", index) + 2;
      final var endKey = json.indexOf("\\\"", beginKey);
      final var key = json.substring(beginKey, endKey);

      final var beginUserData = json.indexOf("{", endKey + 2);
      index = beginUserData + 1;
      for (var openCount = 1; ; ) {
        final var ch = json.charAt(index++);
        if (ch == '{') {
          openCount++;
          continue;
        }
        if (ch == '}' && --openCount == 0) {
          final var userData = json.substring(beginUserData, index).replace("\\\"", "\"");
          idToUserMap.put(key, new User(key, userData));
          break;
        }
      }
      // continue to end or next key/value.
      for (; ; ) {
        final var ch = json.charAt(index++);
        if (ch == '}') {
          return;
        }
        if (ch == ',') {
          break;
        }
      }
    }
  }

  private static String getEscapedJsonStringObject(final String json, final String field) {
    final var userIdIndex = json.indexOf(field);
    if (userIdIndex < 0) {
      return null;
    }
    final var begin = json.indexOf("{", userIdIndex + field.length());
    for (int openCount = 1, index = begin + 1; ; ) {
      final var ch = json.charAt(index++);
      if (ch == '{') {
        openCount++;
        continue;
      }
      if (ch == '}' && --openCount == 0) {
        return json.substring(begin, index);
      }
    }
  }

  @SuppressWarnings("rawtypes")
  private void handleMemberAddedEvent(final String rawJson) {
    final var userId = getEscapedJsonValue(rawJson, "\\\"user_id\\\"");
    final var userInfo = getEscapedJsonStringObject(rawJson, "\\\"user_info\\\"");
    final var user = new User(userId, userInfo.replace("\\\"", "\""));
    idToUserMap.put(userId, user);
    final var listener = getEventListener();
    if (listener != null) {
      final var presenceListener = (PresenceChannelEventListener) listener;
      presenceListener.userSubscribed(getChannelName(), user);
    }
  }

  @SuppressWarnings("rawtypes")
  private void handleMemberRemovedEvent(final String rawJson) {
    final var userId = getEscapedJsonValue(rawJson, "\\\"user_id\\\"");
    final var user = idToUserMap.remove(userId);
    final var listener = getEventListener();
    if (listener != null) {
      final var presenceListener = (PresenceChannelEventListener) listener;
      presenceListener.userUnsubscribed(getChannelName(), user);
    }
  }
}
