package com.pusher.client.channel.impl;

import com.pusher.client.AuthorizationFailureException;
import com.pusher.client.channel.*;
import com.pusher.client.connection.ConnectionEventListener;
import com.pusher.client.connection.ConnectionState;
import com.pusher.client.connection.ConnectionStateChange;
import com.pusher.client.connection.impl.InternalConnection;
import com.pusher.client.util.Factory;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static com.pusher.client.util.PusherJsonParser.getJsonValue;

public class ChannelManager implements ConnectionEventListener {

  private final Map<String, InternalChannel> channelNameToChannelMap = new ConcurrentHashMap<>();
  private final Factory factory;
  private InternalConnection connection;

  public ChannelManager(final Factory factory) {
    this.factory = factory;
  }

  public Channel getChannel(final String channelName) {
    if (channelName.startsWith("private-")) {
      throw new IllegalArgumentException("Please use the getPrivateChannel method");
    }
    if (channelName.startsWith("presence-")) {
      throw new IllegalArgumentException("Please use the getPresenceChannel method");
    }
    return findChannelInChannelMap(channelName);
  }

  public PrivateChannel getPrivateChannel(final String channelName) throws IllegalArgumentException {
    if (!channelName.startsWith("private-")) {
      throw new IllegalArgumentException("Private channels must begin with 'private-'");
    }
    return (PrivateChannel) findChannelInChannelMap(channelName);
  }

  public PresenceChannel getPresenceChannel(final String channelName) throws IllegalArgumentException {
    if (!channelName.startsWith("presence-")) {
      throw new IllegalArgumentException("Presence channels must begin with 'presence-'");
    }
    return (PresenceChannel) findChannelInChannelMap(channelName);
  }

  private InternalChannel findChannelInChannelMap(final String channelName) {
    return channelNameToChannelMap.get(channelName);
  }

  public void setConnection(final InternalConnection connection) {
    if (connection == null) {
      throw new IllegalArgumentException("Cannot construct ChannelManager with a null connection");
    }
    if (this.connection != null) {
      this.connection.unbind(ConnectionState.CONNECTED, this);
    }
    this.connection = connection;
    connection.bind(ConnectionState.CONNECTED, this);
  }

  public void subscribeTo(final InternalChannel channel, final ChannelEventListener listener, final String... eventNames) {
    validateArgumentsAndBindEvents(channel, listener, eventNames);
    channelNameToChannelMap.put(channel.getChannelName(), channel);
    sendOrQueueSubscribeMessage(channel);
  }

  public void unsubscribeFrom(final String channelName) {
    if (channelName == null) {
      throw new IllegalArgumentException("Cannot unsubscribe from null channel");
    }
    final var channel = channelNameToChannelMap.remove(channelName);
    if (channel != null && connection.getState() == ConnectionState.CONNECTED) {
      sendUnsubscribeMessage(channel);
    }
  }

  @SuppressWarnings("unchecked")
  public void onMessage(final String event, final String rawJson) {
    final var channelName = getJsonValue(rawJson, "\"channel\"");
    if (channelName != null) {
      final var channel = channelNameToChannelMap.get(channelName);
      if (channel != null) {
        channel.onMessage(event, rawJson);
      }
    }
  }

  /* ConnectionEventListener implementation */

  @Override
  public void onConnectionStateChange(final ConnectionStateChange change) {
    if (change.getCurrentState() == ConnectionState.CONNECTED) {
      for (final var channel : channelNameToChannelMap.values()) {
        sendOrQueueSubscribeMessage(channel);
      }
    }
  }

  @Override
  public void onError(final String message, final String code, final Throwable e) {
    // ignore or log
  }

  /* implementation detail */

  private void sendOrQueueSubscribeMessage(final InternalChannel channel) {
    factory.queueOnEventThread(() -> {
      if (connection.getState() == ConnectionState.CONNECTED) {
        try {
          connection.sendMessage(channel.toSubscribeMessage());
          channel.updateState(ChannelState.SUBSCRIBE_SENT);
        } catch (final AuthorizationFailureException e) {
          clearDownSubscription(channel, e);
        }
      }
    });
  }

  private void sendUnsubscribeMessage(final InternalChannel channel) {
    factory.queueOnEventThread(() -> {
      connection.sendMessage(channel.toUnsubscribeMessage());
      channel.updateState(ChannelState.UNSUBSCRIBED);
    });
  }

  private void clearDownSubscription(final InternalChannel channel, final Exception e) {
    channelNameToChannelMap.remove(channel.getChannelName());
    channel.updateState(ChannelState.FAILED);
    if (channel.getEventListener() != null) {
      factory.queueOnEventThread(() -> {
        // Note: this cast is safe because an
        // AuthorizationFailureException will never be thrown
        // when subscribing to a non-private channel
        ((PrivateChannelEventListener) channel.getEventListener()).onAuthenticationFailure(e.getMessage(), e);
      });
    }
  }

  private void validateArgumentsAndBindEvents(final InternalChannel channel, final ChannelEventListener listener, final String... eventNames) {
    if (channel == null) {
      throw new IllegalArgumentException("Cannot subscribe to a null channel");
    }
    if (channelNameToChannelMap.containsKey(channel.getChannelName())) {
      throw new IllegalArgumentException("Already subscribed to a channel with channelName " + channel.getChannelName());
    }
    for (final var eventName : eventNames) {
      channel.bind(eventName, listener);
    }
    channel.setEventListener(listener);
  }
}
