package systems.comodal.guava.services;

import com.google.common.util.concurrent.ServiceManager;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Logger;

public final class ServiceRunner {

  private ServiceRunner() {
  }

  public static void main(final String[] args) {
    if (args.length == 0) {
      throw new IllegalStateException("No service group providers provided.");
    }
    awaitHealthy(createServiceGroups(args));
  }

  private static List<ServiceManager> createServiceGroups(final String[] args) {
    final List<ServiceManager> serviceGroups = new ArrayList<>(args.length);
    Arrays.stream(args).map(ServiceRunner::loadClass).map(ServiceRunner::createServiceManager)
        .peek(ServiceManager::startAsync).forEach(serviceGroups::add);
    registerShutdownHook(serviceGroups);
    return serviceGroups;
  }

  private static void awaitHealthy(final List<ServiceManager> serviceGroups) {
    final Logger log = Logger.getLogger(ServiceRunner.class.getName());
    for (final ServiceManager serviceGroup : serviceGroups) {
      serviceGroup.awaitHealthy();
      log.info(serviceGroup.startupTimes().toString());
    }
  }

  private static void registerShutdownHook(final List<ServiceManager> serviceGroups) {
    Runtime.getRuntime()
        .addShutdownHook(new Thread(() -> serviceGroups.forEach(ServiceManager::stopAsync)));
  }

  private static ServiceManager createServiceManager(
      final Class<? extends ServiceGroupLoader> serviceLoaderClass) {
    try {
      final ServiceGroupLoader serviceGroupLoader = serviceLoaderClass.newInstance();
      final ServiceManager serviceManager = new ServiceManager(serviceGroupLoader.loadServices());
      serviceGroupLoader.addListeners(serviceManager);
      return serviceManager;
    } catch (final ReflectiveOperationException e) {
      throw new IllegalStateException(
          serviceLoaderClass + " must have a public zero-argument constructor.");
    }
  }

  @SuppressWarnings("unchecked")
  static Class<? extends ServiceGroupLoader> loadClass(final String className) {
    try {
      final Class<?> cls = Class.forName(className, true, ServiceRunner.class.getClassLoader());
      if (ServiceGroupLoader.class.isAssignableFrom(cls)) {
        return (Class<? extends ServiceGroupLoader>) cls;
      }
      throw new IllegalStateException(
          String.format("Class '%s' does not implement ServiceGroupLoader.", className));
    } catch (final ClassNotFoundException e) {
      throw new IllegalStateException("No class found for " + className);
    }
  }
}
