(ns toolbelt.spec
  "A `spec` library of those commonly used."
  (:require [clojure.spec.alpha :as s]
            [toolbelt.async :as ta]))


;; async helpers ========================


(defn channel
  "Takes a spec and returns a spec for a channel.
  The inner spec is ignored, and used just for documentation purposes."
  ([] (channel any?))
  ([spec] ta/chan?))


(defn async
  "Takes a spec and returns either a spec for the passed-in
  inner spec OR a channel.

  If the Stripe method called is async, the inner spec is ignored and
  used just for documentation purposes. If not, the inner spec is used."
  ([] (async any?))
  ([spec]
   (s/or :spec spec :channel (channel spec))))


;; basic specs ==========================


(s/def ::id string?)

(s/def ::description (s/nilable string?))

(s/def ::pos-float
  (s/or :zero zero? :pos (s/and pos? float?)))

(s/def ::amount
  (s/and pos? number?))


;; email ================================


(def email-regex #"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,63}$")

(s/def ::email (s/and string? #(re-matches email-regex %)))

(defn email? [x] (s/valid? ::email x))


;; timestamp ============================


(s/def ::unix-timestamp
  integer?)

(defn unix-timestamp? [ts]
  (s/valid? ::unix-timestamp ts))


;; math helpers =========================


(defn between
  "Returns a predicate that checks that the supplied number falls
  between the inclusive lower and exclusive upper bounds supplied."
  [low high]
  (fn [x]
    (and (>= x low)
         (< x high))))


;; TODO move this elsewhere?
(defn collectify [x]
  #?(:cljs
     (if (sequential? x) x [x])
     :clj
     (cond (nil? x) []
           (or (sequential? x) (instance? java.util.List x) (set? x)) x
           :else [x])))
