package sqlite4a;

import android.support.annotation.IntRange;
import android.support.annotation.NonNull;

import java.io.Closeable;

/**
 * @author Daniel Serdyukov
 */
@SuppressWarnings("ConstantConditions")
public class SQLiteStmt implements Closeable {

    private final long mStmtPtr;

    SQLiteStmt(long stmtPtr) {
        mStmtPtr = stmtPtr;
    }

    private static native String nativeGetSql(long stmtPtr);

    private static native void nativeClearBindings(long stmtPtr);

    private static native void nativeBindNull(long stmtPtr, int index);

    private static native void nativeBindLong(long stmtPtr, int index, long value);

    private static native void nativeBindString(long stmtPtr, int index, String value);

    private static native void nativeBindDouble(long stmtPtr, int index, double value);

    private static native void nativeBindBlob(long stmtPtr, int index, byte[] value);

    private static native long nativeExecuteInsert(long stmtPtr);

    private static native int nativeExecuteUpdateDelete(long stmtPtr);

    private static native void nativeFinalize(long stmtPtr);

    @NonNull
    public String getSql() {
        return nativeGetSql(mStmtPtr);
    }

    public void clearBindings() {
        nativeClearBindings(mStmtPtr);
    }

    public void bindNull(@IntRange(from = 1) int index) {
        nativeBindNull(mStmtPtr, index);
    }

    public void bindLong(@IntRange(from = 1) int index, long value) {
        nativeBindLong(mStmtPtr, index, value);
    }

    public void bindString(@IntRange(from = 1) int index, @NonNull String value) {
        if (value == null) {
            nativeBindNull(mStmtPtr, index);
        } else {
            nativeBindString(mStmtPtr, index, value);
        }
    }

    public void bindDouble(@IntRange(from = 1) int index, double value) {
        nativeBindDouble(mStmtPtr, index, value);
    }

    public void bindBlob(@IntRange(from = 1) int index, @NonNull byte[] value) {
        if (value == null) {
            nativeBindNull(mStmtPtr, index);
        } else {
            nativeBindBlob(mStmtPtr, index, value);
        }
    }

    @NonNull
    public SQLiteCursor executeQuery() {
        return new SQLiteCursor(mStmtPtr);
    }

    public long executeInsert() {
        return nativeExecuteInsert(mStmtPtr);
    }

    public int executeUpdateDelete() {
        return nativeExecuteUpdateDelete(mStmtPtr);
    }

    @Override
    public void close() {
        nativeFinalize(mStmtPtr);
    }

}
