package sqlite4a;

import android.support.annotation.IntDef;
import android.support.annotation.NonNull;

import java.io.Closeable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Comparator;

/**
 * @author Daniel Serdyukov
 */
public class SQLiteDb implements Closeable {

    public static final int OPEN_READONLY = 0b1;

    public static final int OPEN_READWRITE = 0b10;

    public static final int OPEN_CREATE = 0b100;

    public static final int OPEN_URI = 0b1000000;

    private final long mDbPtr;

    private SQLiteDb(long dbPtr) {
        mDbPtr = dbPtr;
    }

    public static void loadLibrary() {
        Runtime.getRuntime().loadLibrary("sqlite4a");
    }

    @NonNull
    public static native String getLibraryVersion();

    @NonNull
    public static SQLiteDb open(@NonNull String path, @OpenFlags int openFlags) {
        return new SQLiteDb(nativeOpen(path, openFlags));
    }

    private static native long nativeOpen(String path, int openFlags);

    private static native void nativeEnableTracing(long dbPtr);

    private static native void nativeCreateCollation(long dbPtr, String name, Comparator<String> collation);

    private static native void nativeCreateFunction(long dbPtr, String name, int numArgs, SQLiteCustomFunc func);

    private static native void nativeExec(long dbPtr, String sql);

    private static native long nativePrepare(long dbPtr, String sql);

    private static native void nativeClose(long dbPtr);

    public void enableTracing() {
        nativeEnableTracing(mDbPtr);
    }

    public void createCollation(@NonNull String name, @NonNull Comparator<String> collation) {
        nativeCreateCollation(mDbPtr, name, collation);
    }

    public void createFunction(@NonNull String name, int numArgs, @NonNull SQLiteFunc func) {
        nativeCreateFunction(mDbPtr, name, numArgs, new SQLiteCustomFunc(func));
    }

    public void exec(@NonNull String sql) {
        nativeExec(mDbPtr, sql);
    }

    @NonNull
    public SQLiteStmt prepare(@NonNull String sql) {
        return new SQLiteStmt(nativePrepare(mDbPtr, sql));
    }

    @Override
    public void close() {
        nativeClose(mDbPtr);
    }

    @IntDef(flag = true, value = {OPEN_READONLY, OPEN_READWRITE, OPEN_CREATE, OPEN_URI})
    @Retention(RetentionPolicy.SOURCE)
    public @interface OpenFlags {
    }

}
