package sqlite4a;

import android.support.annotation.IntDef;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;

import java.io.Closeable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * @author Daniel Serdyukov
 */
public abstract class SQLiteDb implements Closeable {

    public static final int OPEN_READONLY = 0b1;

    public static final int OPEN_READWRITE = 0b10;

    public static final int OPEN_CREATE = 0b100;

    public static final int OPEN_URI = 0b1000000;

    private static final boolean JDBC_IN_CLASSPATH = SQLiteJdbcDb.inClasspath();

    public static void loadLibrary() {
        if (!JDBC_IN_CLASSPATH) {
            Runtime.getRuntime().loadLibrary("sqlite4a");
        }
    }

    @NonNull
    public static SQLiteDb open(@NonNull String path, @OpenFlags int openFlags) {
        if (JDBC_IN_CLASSPATH) {
            return SQLiteJdbcDb.openConnection(path);
        } else {
            return SQLiteJniDb.openV2(path, openFlags);
        }
    }

    public abstract boolean isReadOnly();

    public abstract boolean isAutoCommit();

    public abstract void trace();

    public abstract void profile(int thresholdMs);

    public abstract void exec(@NonNull String sql);

    @NonNull
    public abstract SQLiteStmt prepare(@NonNull String sql);

    @IntRange(from = 0)
    public abstract int getChanges();

    @IntRange(from = -1)
    public abstract long getLastInsertRowId();

    @Override
    public abstract void close();

    @IntDef(flag = true, value = {OPEN_READONLY, OPEN_READWRITE, OPEN_CREATE, OPEN_URI})
    @Retention(RetentionPolicy.SOURCE)
    public @interface OpenFlags {
    }

}
