package sqlite4a;

import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

/**
 * @author Daniel Serdyukov
 */
class SQLiteJniDb extends SQLiteDb {

    private final long mDbPtr;

    SQLiteJniDb(long dbPtr) {
        mDbPtr = dbPtr;
    }

    @NonNull
    static SQLiteJniDb openV2(@NonNull String path, @OpenFlags int openFlags) {
        return new SQLiteJniDb(nativeOpen(path, openFlags));
    }

    private static native long nativeOpen(String path, int openFlags);

    private static native void nativeTrace(long dbPtr, Logger func);

    private static native void nativeProfile(long dbPtr, Logger func);

    private static native boolean nativeIsReadOnly(long dbPtr);

    private static native boolean nativeIsAutoCommit(long dbPtr);

    private static native void nativeExec(long dbPtr, String sql, Exec func);

    @IntRange(from = 0)
    private static native int nativeGetChanges(long dbPtr);

    @IntRange(from = -1)
    private static native long nativeGetLastInsertRowId(long dbPtr);

    private static native void nativeClose(long dbPtr);

    @Override
    public boolean isReadOnly() {
        return nativeIsReadOnly(mDbPtr);
    }

    @Override
    public boolean isAutoCommit() {
        return nativeIsAutoCommit(mDbPtr);
    }

    @Override
    public void trace(@NonNull Logger func) {
        nativeTrace(mDbPtr, func);
    }

    @Override
    public void profile(@NonNull Logger func) {
        nativeProfile(mDbPtr, func);
    }

    @Override
    public void exec(@NonNull String sql, @Nullable Exec func) {
        nativeExec(mDbPtr, sql, func);
    }

    @NonNull
    @Override
    public SQLiteStmt prepare(@NonNull String sql) {
        return new SQLiteJniStmt(mDbPtr, sql);
    }

    @Override
    public int getChanges() {
        return nativeGetChanges(mDbPtr);
    }

    @Override
    public long getLastInsertRowId() {
        return nativeGetLastInsertRowId(mDbPtr);
    }

    @Override
    public void close() {
        nativeClose(mDbPtr);
    }

}
