package sqlite4a;

import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;

/**
 * @author Daniel Serdyukov
 */
@SuppressWarnings("TryFinallyCanBeTryWithResources")
class SQLiteJdbcStmt implements SQLiteStmt {

    private final String mSql;

    private final PreparedStatement mStmt;

    SQLiteJdbcStmt(@NonNull Connection connection, @NonNull String sql) {
        try {
            mSql = sql;
            mStmt = connection.prepareStatement(sql);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @NonNull
    @Override
    public String getSql() {
        return mSql;
    }

    @Override
    public void reset() {

    }

    @Override
    public void clearBindings() {
        try {
            mStmt.clearParameters();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public void bindNull(@IntRange(from = 1) int index) {
        try {
            mStmt.setNull(index, Types.OTHER);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public void bindLong(@IntRange(from = 1) int index, long value) {
        try {
            mStmt.setLong(index, value);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public void bindDouble(@IntRange(from = 1) int index, double value) {
        try {
            mStmt.setDouble(index, value);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public void bindString(@IntRange(from = 1) int index, @NonNull String value) {
        try {
            mStmt.setString(index, value);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public void bindBlob(@IntRange(from = 1) int index, @NonNull byte[] value) {
        try {
            mStmt.setBytes(index, value);
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public int execute() {
        try {
            return mStmt.executeUpdate();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @NonNull
    @Override
    public SQLiteRowSet executeSelect() {
        try {
            return new SQLiteJdbcRowSet(mStmt.executeQuery());
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @Override
    public boolean isBusy() {
        return false;
    }

    @Override
    public void close() {
        try {
            mStmt.close();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @VisibleForTesting
    void execute(@Nullable SQLiteDb.Exec func) {
        try {
            final String sql = mSql.toLowerCase();
            if (sql.startsWith("begin")) {
                mStmt.getConnection().setAutoCommit(false);
            } else if (sql.startsWith("commit")) {
                mStmt.getConnection().commit();
                mStmt.getConnection().setAutoCommit(true);
            } else if (sql.startsWith("rollback")) {
                mStmt.getConnection().rollback();
                mStmt.getConnection().setAutoCommit(true);
            } else {
                executeStatement(func);
            }
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @VisibleForTesting
    int getChanges() {
        try {
            return mStmt.getUpdateCount();
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    @VisibleForTesting
    long getLastInsertRowId() {
        try {
            final ResultSet rs = mStmt.getGeneratedKeys();
            try {
                if (rs.next()) {
                    return rs.getLong(1);
                }
            } finally {
                rs.close();
            }
            return -1;
        } catch (SQLException e) {
            throw new SQLiteException(e);
        }
    }

    private void executeStatement(@Nullable SQLiteDb.Exec func) throws SQLException {
        if (func == null) {
            mStmt.execute();
        } else {
            final ResultSet rs = mStmt.executeQuery();
            try {
                final ResultSetMetaData metaData = rs.getMetaData();
                final int columnCount = metaData.getColumnCount();
                final String[] columns = new String[columnCount];
                for (int columnIndex = 0; columnIndex < columnCount; ++columnIndex) {
                    columns[columnIndex] = metaData.getColumnName(columnIndex + 1);
                }
                while (rs.next()) {
                    final String[] values = new String[columnCount];
                    for (int columnIndex = 0; columnIndex < columnCount; ++columnIndex) {
                        values[columnIndex] = rs.getString(columnIndex + 1);
                    }
                    func.call(values, columns);
                }
            } finally {
                rs.close();
            }
        }
    }

}
